"use strict";

import {store, actions} from './model/user_model.js';
import {Usermgmt} from './component/usermgmt_component.js';
import {handleDropDownOnClickById, displayDropdownBoxLabelById, onNavInitialize} from './common.js';
import {hasInvalidCharacter, isValidEmail, isValidPassword, handleInputLabels} from './common.js';
import {checkRequiredFieldInItem, checkInvalidCharacterInItem} from './common.js';
import {checkInvalidPasswordInItem, checkInvalidEmailInItem} from './common.js';
import {convertDatetimeToLocaleDate, DateDiff} from './common.js';
import {camelCaseToSentenceCase} from './common.js';
import {handleViewPasswordIconOnClick} from './common.js';
import {getReorderedHeadersAndRecArrs} from './component/history_export.js';

const usermgmt = Usermgmt(store, actions);

usermgmt.buildUsermgmtQuery = () => {
  // Set event for query param
  const {userGroups, departments} = store.getState();
  const form = document.getElementById("usermgmt-query-form");

  const usernameInput = form.querySelector("[data-usermgmt-query='username']").querySelector("input");
  usernameInput.addEventListener('input', (e) => usermgmt.handleQueryInputOnChange('username', e.target.value));

  const deptSelect = form.querySelector("[data-usermgmt-query='departmentCode']").querySelector("select");
  departments.forEach(dept => {
    const child = document.createElement('option');
    child.value = dept.departmentCode;
    child.innerHTML = dept.departmentName;
    deptSelect.appendChild(child);
  });
  deptSelect.selectedIndex = -1;
  deptSelect.addEventListener('change', (e) => usermgmt.handleQueryInputOnChange('departmentCode', e.target.value));

  const groupSelect = form.querySelector("[data-usermgmt-query='groupId']").querySelector("select");
  userGroups.forEach(g => {
    const child = document.createElement('option');
    child.value = g.id;
    child.innerHTML = g.groupName;
    groupSelect.appendChild(child);
  });
  groupSelect.selectedIndex = -1;
  groupSelect.addEventListener('change', (e) => usermgmt.handleQueryInputOnChange('groupId', e.target.value));

  const statusSelect = form.querySelector('#usermgmt-query-status');
  statusSelect.querySelector("[data-dropdown='box']").addEventListener('click', () => handleDropDownOnClickById('usermgmt-query-status'));
  statusSelect.querySelector("[data-dropdown='mask']").addEventListener('click', () => handleDropDownOnClickById('usermgmt-query-status'));
  const statusInputs = statusSelect.querySelector("[data-dropdown='container']").querySelectorAll('input');
  statusInputs.forEach(input => {
    input.addEventListener('change', () => displayDropdownBoxLabelById(form, 'usermgmt-query-status'));
    input.addEventListener('change', () => usermgmt.handleQueryInputOnChange('status', Array.from(statusInputs).filter(i => i.checked).map(o => o.value)));
  });
};

usermgmt.clearUsermgmtQuery = () => {
  const form = document.getElementById("usermgmt-query-form");

  const usernameInput = form.querySelector("[data-usermgmt-query='username']").querySelector("input");
  usernameInput.value = "";

  const deptSelect = form.querySelector("[data-usermgmt-query='departmentCode']").querySelector("select");
  deptSelect.selectedIndex = -1;

  const groupSelect = form.querySelector("[data-usermgmt-query='groupId']").querySelector("select");
  groupSelect.selectedIndex = -1;

  const statusSelect = form.querySelector('#usermgmt-query-status');
  const statusInputs = statusSelect.querySelector("[data-dropdown='container']").querySelectorAll('input');
  statusInputs.forEach(input => {
    input.checked = false;
  });
  const boxLabel = statusSelect.querySelector("[data-dropdown='box']").querySelector('p');
  boxLabel.innerHTML = "";
};

usermgmt.fillUsermgmtTableRow = (cols, item, viewButton) => {
  const {userGroups, departments} = store.getState();
  const groupNames = {};
  userGroups.forEach(g => {
    groupNames[g.id] = g.groupName
  });
  // Username group title status
  cols[0].textContent = item.username;
  let statusName;
  if (item.status === 1) {
    statusName = 'active'; 
  } else if (item.status === 9) {
    statusName = 'locked'; 
  } else if (item.status === 0) {
    statusName = 'disabled'; 
  }
  const statusSpan = document.getElementById("usermgmt-status-labels").content.querySelector(`[data-usermgmt-status='${statusName}']`).cloneNode(true);
  cols[1].appendChild(statusSpan);

  const dept = departments.find(d => d.departmentCode === item.departmentCode);
  const groupNameText = item.groupIds && item.groupIds.length ? item.groupIds.map(id => groupNames[id]).reduce((prev, cur) => `${prev}, ${cur}`) : '';
  cols[2].textContent = dept && dept.departmentName ? dept.departmentName : '';
  cols[3].textContent = groupNameText;
  cols[4].appendChild(viewButton);
  cols[4].classList.add('text-right');
};

usermgmt.renderViewItemContent = (formTable, itm) => {
  // Handle form
  const {departments, userGroups} = store.getState();

  // username
  const usernameInput = formTable.querySelector("[data-usermgmt-detail='username']").querySelector("input");
  usernameInput.value = itm.username ? itm.username : null;
  usernameInput.disabled = true;

  // new password
  const pwContainer = formTable.querySelector("[data-usermgmt-detail='password']");
  const pwInput = pwContainer.querySelector("input");
  pwInput.disabled = true;
  pwInput.addEventListener('input', (e) => usermgmt.handleFormInputOnChange('password', e.target.value));
  // pwInput.addEventListener('input', (e) => {
  //   if (e.target.type !== 'password') {
  //     e.target.type = 'password';
  //   }
  // });
  const viewpwIcons = document.querySelector("[data-login-form='viewpw']")
  const showpwIcon = viewpwIcons.querySelector("[data-form-viewpw='show']");
  const hidepwIcon = viewpwIcons.querySelector("[data-form-viewpw='hide']");
  showpwIcon.addEventListener("click", () => handleViewPasswordIconOnClick(true, pwInput, showpwIcon, hidepwIcon));
  hidepwIcon.addEventListener("click", () => handleViewPasswordIconOnClick(false, pwInput, showpwIcon, hidepwIcon));

  const pwExpiryContainer = formTable.querySelector("[data-usermgmt-detail='passwordExpiryDate']");
  if (itm.passwordExpiryDate === null || itm.passwordExpiryDate === undefined) {
    pwExpiryContainer.remove();
  } else {
    const renderExpiryDate = (container, expiryDate) => {
      // Handle input style
      const pwExpirySpan = container.querySelector("span");
      const d = new Date(expiryDate);
      const today = new Date();
      const diff = DateDiff.inDays(today, d);
      const displayDate = convertDatetimeToLocaleDate(expiryDate);
      if (diff >= 3) {
        pwExpirySpan.textContent = `${displayDate}`;
        pwExpirySpan.classList.add('text-green-600');
      } else if (diff < 3 && diff >= 0) {
        pwExpirySpan.textContent = `${displayDate} (Expire Soon)`;
        pwExpirySpan.classList.add('text-yellow-600');
      } else {
        pwExpirySpan.textContent = `${displayDate} (Expired)`;
        pwExpirySpan.classList.add('text-red-600');
      }
    };
    renderExpiryDate(pwExpiryContainer, itm.passwordExpiryDate);
  }

  const isNewPwCell = formTable.querySelector("[data-usermgmt-detail='isNewPassword']");
  Array.from(isNewPwCell.children).forEach(child => {
    if (!child.hasAttribute("data-usermgmt-detail-isnewpw")) {
      child.remove();
    }
  });
  const isNewPwCheckbox = isNewPwCell.querySelector("input");
  isNewPwCheckbox.addEventListener('click', () => handleIsNewPwCheckboxOnClick(isNewPwCheckbox, pwContainer));

  // id
  const idInput = formTable.querySelector("[data-usermgmt-detail='id']").querySelector("input");
  idInput.value = itm.id ? itm.id : null;
  idInput.disabled = true;

  // status
  const statusParts = formTable.querySelector("[data-usermgmt-detail='status']");
  const statusActive = statusParts.querySelector("[data-usermgmt-status='active']");
  const statusActiveBtn = statusActive.querySelector("Button");
  const statusDisabled = statusParts.querySelector("[data-usermgmt-status='disabled']");
  const statusDisabledBtn = statusDisabled.querySelector("Button");
  const statusLocked = statusParts.querySelector("[data-usermgmt-status='locked']");
  const statusLockedBtn = statusLocked.querySelector("Button");
  statusActiveBtn.addEventListener('click', (e) => handleStatusButtonOnClick(e, itm.id, 0));
  statusDisabledBtn.addEventListener('click', (e) => handleStatusButtonOnClick(e, itm.id, 1));
  statusLockedBtn.addEventListener('click', (e) => handleStatusButtonOnClick(e, itm.id, 1));
  if (itm.status === 0) {
    statusActive.classList.add('hidden');
    statusDisabled.classList.remove('hidden');
    statusLocked.classList.add('hidden');
  } else if (itm.status === 1) {
    statusActive.classList.remove('hidden');
    statusDisabled.classList.add('hidden');
    statusLocked.classList.add('hidden');
  } else if (itm.status === 9) {
    statusActive.classList.add('hidden');
    statusDisabled.classList.add('hidden');
    statusLocked.classList.remove('hidden');
  } else {
    console.error("Unknown itm status");
  }

  // email 
  const emailInput = formTable.querySelector("[data-usermgmt-detail='email']").querySelector("input");
  emailInput.value = itm.email ? itm.email : null;
  emailInput.addEventListener('input', (e) => usermgmt.handleFormInputOnChange('email', e.target.value));

  // departments
  const deptSelect = formTable.querySelector("[data-usermgmt-detail='departmentCode']").querySelector("select");
  departments.forEach(dept => {
    const child = document.createElement('option');
    child.value = dept.departmentCode;
    child.innerHTML = `${dept.departmentName}  [${dept.departmentCode}]`;
    if (dept.departmentCode === itm.departmentCode) {
      child.selected = true;
    }
    deptSelect.appendChild(child);
  });
  deptSelect.addEventListener('change', (e) => usermgmt.handleFormInputOnChange('departmentCode', e.target.value));

  const groupSelect = formTable.querySelector("#usermgmt-user-groups");
  groupSelect.querySelector("[data-dropdown='box']").addEventListener('click', () => handleDropDownOnClickById('usermgmt-user-groups'));
  groupSelect.querySelector("[data-dropdown='mask']").addEventListener('click', () => handleDropDownOnClickById('usermgmt-user-groups'));
  const groupList = groupSelect.querySelector("[data-dropdown='container']").querySelector("ul");
  const dropdownItemTemplate = formTable.querySelector("[data-usermgmt-detail='groupIds']").querySelector("template");
  userGroups.forEach((g, idx) => {
    const option = dropdownItemTemplate.cloneNode(true).content;
    const input = option.querySelector("input");
    input.value = g.id;
    option.querySelector("span").textContent = g.groupName;
    if (itm.groupIds.findIndex(id => id === g.id) >= 0) {
      input.checked = true;
    }
    groupList.appendChild(option);
  });
  const groupInputs = groupList.querySelectorAll("input");
  groupInputs.forEach(input => {
    input.addEventListener('change', () => displayDropdownBoxLabelById(formTable, 'usermgmt-user-groups'));
    input.addEventListener('change', () => handleFormMultiSelectDropdownOnChange('groupIds', Array.from(groupInputs).filter(i => i.checked).map(o => o.value)));
  });
  displayDropdownBoxLabelById(formTable, 'usermgmt-user-groups');

};

usermgmt.renderAddItemContent = (formTable) => {
  const {departments, userGroups} = store.getState();

  // username
  const usernameInput = formTable.querySelector("[data-usermgmt-detail='username']").querySelector("input");
  usernameInput.required = true;
  usernameInput.addEventListener('input', (e) => usermgmt.handleFormInputOnChange('username', e.target.value));

  // password
  const pwInput = formTable.querySelector("[data-usermgmt-detail='password']").querySelector("input");
  pwInput.addEventListener('input', (e) => usermgmt.handleFormInputOnChange('password', e.target.value));
  const viewpwIcons = document.querySelector("[data-login-form='viewpw']")
  const showpwIcon = viewpwIcons.querySelector("[data-form-viewpw='show']");
  const hidepwIcon = viewpwIcons.querySelector("[data-form-viewpw='hide']");
  showpwIcon.addEventListener("click", () => handleViewPasswordIconOnClick(true, pwInput, showpwIcon, hidepwIcon));
  hidepwIcon.addEventListener("click", () => handleViewPasswordIconOnClick(false, pwInput, showpwIcon, hidepwIcon));
  // pwInput.addEventListener('input', (e) => {
  //   if (e.target.type !== 'password') {
  //     e.target.type = 'password';
  //   }
  // });
  pwInput.required = true;
  const isNewPwCell = formTable.querySelector("[data-usermgmt-detail='isNewPassword']");
  Array.from(isNewPwCell.children).forEach(child => {
    if (child.hasAttribute("data-usermgmt-detail-isnewpw")) {
      child.remove();
    }
  });
  const pwExpiryContainer = formTable.querySelector("[data-usermgmt-detail='passwordExpiryDate']");
  if (pwExpiryContainer) {
    pwExpiryContainer.remove();
  }

  // id
  const idCell = formTable.querySelector("[data-usermgmt-detail='id']");
  idCell.remove();

  const statusParts = formTable.querySelector("[data-usermgmt-detail='status']");
  statusParts.remove();

  // email 
  const emailInput = formTable.querySelector("[data-usermgmt-detail='email']").querySelector("input");
  emailInput.addEventListener('input', (e) => usermgmt.handleFormInputOnChange('email', e.target.value));

  // departments
  const deptSelect = formTable.querySelector("[data-usermgmt-detail='departmentCode']").querySelector("select");
  departments.forEach(dept => {
    const child = document.createElement('option');
    child.value = dept.departmentCode;
    child.innerHTML = `${dept.departmentName}  [${dept.departmentCode}]`;
    deptSelect.appendChild(child);
  });
  deptSelect.addEventListener('change', (e) => usermgmt.handleFormInputOnChange('departmentCode', e.target.value));

  const groupSelect = formTable.querySelector("#usermgmt-user-groups");
  groupSelect.querySelector("[data-dropdown='box']").addEventListener('click', () => handleDropDownOnClickById('usermgmt-user-groups'));
  groupSelect.querySelector("[data-dropdown='mask']").addEventListener('click', () => handleDropDownOnClickById('usermgmt-user-groups'));
  const groupList = groupSelect.querySelector("[data-dropdown='container']").querySelector("ul");
  const dropdownItemTemplate = formTable.querySelector("[data-usermgmt-detail='groupIds']").querySelector("template");
  userGroups.forEach((g, idx) => {
    const option = dropdownItemTemplate.cloneNode(true).content;
    const input = option.querySelector("input");
    input.value = g.id;
    option.querySelector("span").textContent = g.groupName;
    groupList.appendChild(option);
  });
  const groupInputs = groupList.querySelectorAll("input");
  groupInputs.forEach(input => {
    input.addEventListener('change', () => displayDropdownBoxLabelById(formTable, 'usermgmt-user-groups'));
    input.addEventListener('change', () => handleFormMultiSelectDropdownOnChange('groupIds', Array.from(groupInputs).filter(i => i.checked).map(o => o.value)));
  });
};

usermgmt.isAddItemButtonActive = () => {
  const {changeItem} = store.getState();
  let isValid = checkRequiredFieldInItem(changeItem, ['username', 'password']);
  isValid = isValid && checkInvalidCharacterInItem(changeItem, ['username']);
  isValid = isValid && checkInvalidPasswordInItem(changeItem, ['password']);
  isValid = isValid && checkInvalidEmailInItem(changeItem, ['email']);
  return isValid;
};

usermgmt.isUpdateItemButtonActive = () => {
  const {changeItem, items} = store.getState();
  const item = items.find(itm => itm.id === changeItem.id);
  let isActive = false;
  if (item) {
    Object.keys(changeItem).forEach(k => {
      if (changeItem[k] !== null && changeItem[k] !== item[k]) {
        isActive = true;
      }
    });
  }
  if (!isActive) {
    alert("No updated field")
    return isActive;
  }
  isActive = isActive && checkInvalidPasswordInItem(changeItem, ['password']);
  isActive = isActive && checkInvalidEmailInItem(changeItem, ['email']);
  return isActive;
};

usermgmt.isSearchButtonActive = () => {
  const {queryParams} = store.getState();
  return checkInvalidCharacterInItem(queryParams, ['username']);
};

usermgmt.exportRecords = () => {
  const {items, userGroups, departments} = store.getState();
  // Deep copy
  const records = JSON.parse(JSON.stringify(items));
  const departmentsMap = {};
  const userGroupsMap = {};
  const statusMap = {
    0: 'disabled',
    1: 'active',
    9: 'locked',
  };
  departments.forEach(dept => {
    departmentsMap[dept.departmentCode] = dept.departmentName;
  });
  userGroups.forEach(g => {
    userGroupsMap[g.id] = g.groupName;
  });
  records.forEach(rec => {
    rec.status = statusMap[rec.status];
    rec.departmentName = rec.departmentCode ? departmentsMap[rec.departmentCode] : null;
    const groupNames = rec.groupIds.map(id => userGroupsMap[id]);
    rec.groupNames = groupNames.join('|');
    delete rec.groupIds;
    delete rec.password;
  });
  const order = ['id', 'username', 'email', 'status', 'groupNames',
                 'departmentCode', 'departmentName', 'passwordExpiryDate'];
  let {headers, recArrs} = getReorderedHeadersAndRecArrs(records, order)
  headers = headers.map(camelCaseToSentenceCase);
  return {filename: 'user_export.csv', headers, recArrs};
};

const handleIsNewPwCheckboxOnClick = (element, pwContainer) => {
  const pwInput = pwContainer.querySelector("input");
  pwInput.disabled = !element.checked;
  if (!element.checked) {
    pwInput.value = null;
    usermgmt.handleFormInputOnChange('password', null);
  }
  handleInputLabels(pwContainer);
};

const handleFormMultiSelectDropdownOnChange = (name, selected) => {
  store.dispatch(actions.setChangeItem({[name]: selected}));
};

const handleStatusButtonOnClick = (e, id, status) => {
  e.preventDefault();
  let confirmation = confirm("Are you sure to update status?");
  if (confirmation) {
    store.dispatch(actions.updateStatus(id, status));
  }
}

store.subscribe((action) => {
  if (action.type === 'SET_INIT_ITEMS') {
    const {currentPage, itemPerPage, items, currentPageItems} = store.getState();
    usermgmt.buildUsermgmtLayout();
    usermgmt.buildUsermgmtTable(currentPageItems);
    usermgmt.buildUsermgmtPagination(currentPage, itemPerPage, items.length);
  } else if (action.type === 'SET_ITEMS' || action.type === 'SET_PAGE') {
    const {currentPage, itemPerPage, items, currentPageItems, changeItem} = store.getState();
    usermgmt.buildUsermgmtTable(currentPageItems);
    usermgmt.buildUsermgmtPagination(currentPage, itemPerPage, items.length);
    if (usermgmt.isViewing()) {
      const updatedChangeItem = items.find(itm => itm.id === changeItem.id);
      usermgmt.renderViewItem(updatedChangeItem);
    }
  } else if (action.type === 'CLEAR_QUERY') {
    usermgmt.clearUsermgmtQuery();
  }
});

document.addEventListener("DOMContentLoaded", function(){
  store.dispatch(actions.initItems());
  onNavInitialize();
});

