"use strict";

import {createStore, convertUrlQueryParams, getChangedItem, displayActionResponse} from '../common.js';
import {APIPrefix, getJsonRequestHeaderWithCsrf} from '../common.js';

// Redux architecture pieces
export const actions = {
  initItems: () => {
    return {
      type: 'INIT_ITEMS'
    };
  },
  setInitItems: (items) => {
    // To be called by getItems callback
    return {
      type: 'SET_INIT_ITEMS',
      payload: items,
    };
  },
  setUserGroups: (items) => {
    // To be called by getItems callback
    return {
      type: 'SET_USER_GROUPS',
      payload: items,
    };
  },
  setDepartments: (items) => {
    // To be called by getItems callback
    return {
      type: 'SET_DEPARTMENTS',
      payload: items,
    };
  },
  getItems: () => {
    // Get items from fetch
    return {
      type: 'GET_ITEMS'
    };
  },
  setItems: (items) => {
    // To be called by getItems callback
    return {
      type: 'SET_ITEMS',
      payload: items,
    };
  },
  setPage: (page = 0) => {
    // Set currentPage & currentPageItems
    return {
      type: 'SET_PAGE',
      payload: page,
    };
  },
  setQueryParams: (params) => {
    // Set Query params for getting items
    // Make callback to getItems
    return {
      type: 'SET_QUERY',
      payload: params,
    };
  },
  clearQueryParams: () => {
    return {
      type: 'CLEAR_QUERY',
    };
  },
  setChangeItem: (item) => {
    // Set modified item values to store
    // To be submitted to backend
    return {
      type: 'SET_CHANGE_ITEM',
      payload: item,
    };
  },
  clearChangeItem: () => {
    return {
      type: 'CLEAR_CHANGE_ITEM',
    };
  },
  updateItem: () => {
    return {
      type: 'UPDATE_ITEM',
    };
  },
  updateStatus: (id, status) => {
    return {
      type: 'UPDATE_STATUS',
      payload: {
        id,
        status,
      }
    };
  },
  deleteItem: () => {
    return {
      type: 'DELETE_ITEM',
    };
  },
  createItem: () => {
    return {
      type: 'CREATE_ITEM',
    };
  },
};

const initialState = {
  queryParams: {},
  items: [],
  userGroups: [],
  departments: [],
  currentPage: 1,
  itemPerPage: 10,
  currentPageItems: [],
  changeItem: {},
};


const userReducer = (state = initialState, action) => {
  let changeItem, item;
  switch (action.type) {
    case 'INIT_ITEMS':
      initUserItems();
      return state;

    case 'SET_INIT_ITEMS':
      // Trigger layout init
      return Object.assign(Object.assign({}, state), {
        items: action.payload,
        currentPage: 1,
        currentPageItems: action.payload.slice(0, state.itemPerPage)
      });

    case 'SET_USER_GROUPS':
      return Object.assign(Object.assign({}, state), {
        userGroups: action.payload,
      });

    case 'SET_DEPARTMENTS':
      return Object.assign(Object.assign({}, state), {
        departments: action.payload,
      });

    case 'GET_ITEMS':
      getUserItems(state.queryParams);
      return state;

    case 'SET_ITEMS':
      // Set items & initialize page + currentPageItems
      return Object.assign(Object.assign({}, state), {
        items: action.payload,
        currentPage: 1,
        currentPageItems: action.payload.slice(0, state.itemPerPage)
      });

    case 'SET_PAGE':
      const page = action.payload;
      const sIdx = (page - 1) * state.itemPerPage;
      const eIdx = (page) * state.itemPerPage; // Slice does not include eIdx
      return Object.assign(Object.assign({}, state), {
        currentPage: page,
        currentPageItems: state.items.slice(sIdx, eIdx),
      });

    case 'SET_QUERY':
      let queryParams = Object.assign(
         Object.assign({}, state.queryParams), action.payload);
      return Object.assign(Object.assign({}, state), {queryParams});

    case 'CLEAR_QUERY':
      return Object.assign(Object.assign({}, state), {queryParams: {}, page: 0});

    case 'SET_CHANGE_ITEM':
      changeItem = Object.assign(
         Object.assign({}, state.changeItem), action.payload);
      return Object.assign(Object.assign({}, state), {changeItem});

    case 'CLEAR_CHANGE_ITEM':
      return Object.assign(Object.assign({}, state), {changeItem: {}});

    case 'UPDATE_ITEM':
      item = state.items.find(itm => itm.id === state.changeItem.id);
      updateUser(Object.assign(Object.assign({}, item), state.changeItem));
      return state;

    case 'UPDATE_STATUS':
      item = state.items.find(itm => itm.id === action.payload.id);
      updateUserStatus(Object.assign(Object.assign({}, item), {status: action.payload.status}));
      return state;

    case 'DELETE_ITEM':
      deleteUser(state.changeItem.id);
      return state;

    case 'CREATE_ITEM':
      postUser(state.changeItem);
      return state;

    default:
      return state;
  }
};

export const store = createStore(userReducer);

const initUserItems = () => {
  Promise.all([
    fetch(`${APIPrefix}/group`),
    fetch(`${APIPrefix}/department`),
    fetch(`${APIPrefix}/user`),
  ]).then(responses => {
    return Promise.all(responses.map(resp => {
      if (resp.redirected) {
        window.location.href = resp.url;
      } else if (resp.status === 403) {
        alert('Your session has expired. Please login again.');
        window.location.href = '/login';
      } else {
        return resp.json();
      }
    }));
  }).then(results => {
    let data = [];
    results.forEach(result => {
      const errMsg = result && result.resp_msg ? result.resp_msg : '';
      if (result.result === "Success") {
        data.push(result.data);
      } else {
        alert(`${result.result} ${errMsg}`);
        throw new Error("Failed to get data from response json");
      }
    });
    let group = data[0];
    let dept = data[1];
    let user = data[2];
    store.dispatch(actions.setUserGroups(group));
    store.dispatch(actions.setDepartments(dept));
    store.dispatch(actions.setInitItems(user));
  }).catch(error => {
    console.error(error);
  });
};

const getUserItems = (params) => {
  const url = convertUrlQueryParams(`${APIPrefix}/user`, params);
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      store.dispatch(actions.setItems(result.data));
    } else {
      displayActionResponse(result);
    }
  }).catch(error => {
    console.error(error);
  });
};

const postUser = (changeItem) => {
  const body = Object.assign({}, changeItem);
  body.status = 1;
  fetch(`${APIPrefix}/user`, {
    method: 'POST',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const deleteUser = (id) => {
  fetch(`${APIPrefix}/user/${id}`, {
    method: 'DELETE',
    headers: getJsonRequestHeaderWithCsrf(),
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const updateUser = (body) => {
  const id = body.id;
  fetch(`${APIPrefix}/user/${id}`, {
    method: 'PUT',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const updateUserStatus = (body) => {
  const id = body.id;
  fetch(`${APIPrefix}/user/${id}/status`, {
    method: 'PUT',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};
