"use strict";

import {createStore, displayActionResponse, handleDownloadResponse} from '../common.js';
import {APIPrefix, getRequestHeaderWithCsrf, getJsonRequestHeaderWithCsrf} from '../common.js';
import {convertUrlQueryParams} from '../common.js';

export const getDetailTypeFromMessageType = (messageType) => {
  const DetailTypes = {
    'acc.sta.nti.01': 'account-status-change-notification',
    'acc.opa.nti.01': 'account-operation-state-change-notification',
    'cert.exp.nti.01': 'certificate-expiry-notification',
    'clsr.rnw.nti.01': 'client-secret-renewal-notification',
    'sigk.exp.nti.01': 'message-signing-key-expiry-notification',
    'crp.sys.nti.01': 'crp-system-notification',
  };
  return DetailTypes[messageType] ? DetailTypes[messageType] : null;
};

// Redux architecture pieces
export const actions = {
  // Actions for History Page
  setPage: (page) => {
    return {
      type: 'SET_PAGE',
      payload: page
    };
  },
  setSize: (size) => {
    return {
      type: 'SET_SIZE',
      payload: size
    };
  },
  setTotalPages: (totalPages) => {
    return {
      type: 'SET_TOTAL_PAGES',
      payload: totalPages
    };
  },
  setTotalRecords: (totalRecords) => {
    return {
        type: 'SET_TOTAL_RECORDS',
        payload: totalRecords
    };
  },
  getRecords: () => {
    return {
      type: 'GET_RECORDS',
    };
  },
  setRecords: (records) => {
    return {
      type: 'SET_RECORDS',
      payload: records,
    };
  },
  setRecordLoading: (isLoading) => {
    return {
      type: 'SET_RECORD_LOADING',
      payload: isLoading,
    };
  },
  getDetail: (messageId) => {
    return {
      type: 'GET_DETAIL',
      payload: {messageId},
    };
  },
  setDetail: (detail) => {
    return {
      type: 'SET_DETAIL',
      payload: detail
    };
  },
  setDetailLoading: (isLoading) => {
    return {
      type: 'SET_DETAIL_LOADING',
      payload: isLoading,
    };
  },
  setQuery: (param) => {
    return {
      type: 'SET_QUERY',
      payload: param
    };
  },
  clearQuery: () => {
    return {
      type: 'CLEAR_QUERY',
    };
  },
};

const initialState = {
  tag: 'system',
  tab: 'history',
  page: 0,
  totalPages: null,
  totalRecords: null,
  size: 10,
  records: [],
  recordLoading: false,
  detail: {},
  detailLoading: false,
  query: {},
};

const systemNotificationReducer = (state = initialState, action) => {
  let reduce;
  switch (action.type) {
    case 'SET_PAGE':
      reduce = (state, action) => {
        const {size, query} = state;
        getSystemHistory(action.payload, size, query);
        return Object.assign(Object.assign({}, state), {
          page: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_SIZE':
      reduce = (state, action) => {
        const { query } = state;
        const newPage = 0;
        let newSize;
        if (action.payload === null || action.payload === undefined) {
          newSize = initialState.size;
        } else {
          newSize = action.payload;
        }
        getSystemHistory(newPage, newSize, query);
        return Object.assign(Object.assign({}, state), {
          page: newPage,
          size: newSize,
        });
      };
      return reduce(state, action);


    case 'SET_TOTAL_PAGES':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalPages: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalRecords: action.payload,
        });
      };
      return reduce(state, action);

    case 'GET_RECORDS':
      // TODO: handle query param
      reduce = (state, action) => {
        const {page, size, query} = state;
        getSystemHistory(page, size, query);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          records: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_RECORD_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          recordLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_QUERY':
      reduce = (state, action) => {
        const queryItem = Object.assign({}, state.query);
        return Object.assign(Object.assign({}, state), {
          query: Object.assign(queryItem, action.payload),
        });
      };
      return reduce(state, action);

    case 'CLEAR_QUERY':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {query: {}});
      };
      return reduce(state, action);

    case 'GET_DETAIL':
      getSystemDetail(action.payload.messageId);
      return store.getState();

    case 'SET_DETAIL':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detail: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_DETAIL_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detailLoading: action.payload,
        });
      };
      return reduce(state, action);

    default:
      return state;
  }
};

export const store = createStore(systemNotificationReducer);

const getSystemHistory = (page, size, params = {}) => {
  store.dispatch(actions.setRecordLoading(true));
  const url = convertUrlQueryParams(`${APIPrefix}/notification-centre/system/history`, Object.assign({page, size}, params));
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    const {tab} = store.getState();
    if (tab !== 'history') {
      return;
    }
    if (result.result === "Success") {
      store.dispatch(actions.setRecords(result.data.record ? result.data.record : []));
      store.dispatch(actions.setTotalPages(result.data.totalPages));
      store.dispatch(actions.setTotalRecords(result.data.total));
      if (result.data && result.data.total === 0) {
        alert('No matched record found');
      }
    } else {
      displayActionResponse(result);
    }
    store.dispatch(actions.setRecordLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const getSystemDetail = (messageId) => {
  GetSystemDetail(store, actions, messageId);
};

const GetSystemDetail = (curStore, curActions, messageId) => {
  curStore.dispatch(curActions.setDetailLoading(true));
  const url = `${APIPrefix}/notification-centre/system/notification/${messageId}/raw-message`;
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      if (!!result.data.rawMessage) {
        const detail = JSON.stringify(result.data.rawMessage, null, '    ')
        curStore.dispatch(curActions.setDetail(detail));
      } else {
        curStore.dispatch(curActions.setDetail(null));
      }
    } else {
      displayActionResponse(result);
    }
    curStore.dispatch(curActions.setDetailLoading(false));
  }).catch(error => {
    console.error(error);
  });
};


