"use strict";

import {createStore, displayActionResponse} from '../common.js';
import {APIPrefix, getJsonRequestHeaderWithCsrf, convertUrlQueryParams} from '../common.js';
import {handleDownloadResponse} from '../common.js';

// Redux architecture pieces
export const actions = {
  // Actions for Data submission common
  setTab: (tab) => {
    return {
      type: 'SET_TAB',
      payload: tab,
    };
  },
  setSize: (size) => {
    return {
      type: 'SET_SIZE',
      payload: size
    };
  },
  // Actions for History Page
  setPage: (page) => {
    return {
      type: 'SET_PAGE',
      payload: page
    };
  },
  setTotalPages: (totalPages) => {
    return {
      type: 'SET_TOTAL_PAGES',
      payload: totalPages
    };
  },
  setTotalRecords: (totalRecords) => {
    return {
      type: 'SET_TOTAL_RECORDS',
      payload: totalRecords
    };
  },
  getRecords: () => {
    return {
      type: 'GET_RECORDS',
    };
  },
  setRecords: (records) => {
    return {
      type: 'SET_RECORDS',
      payload: records,
    };
  },
  setRecordLoading: (isLoading) => {
    return {
      type: 'SET_RECORD_LOADING',
      payload: isLoading,
    };
  },
  getDetail: (idType, id) => {
    return {
      type: 'GET_DETAIL',
      payload: {idType, id},
    };
  },
  setDetail: (detail) => {
    return {
      type: 'SET_DETAIL',
      payload: detail
    };
  },
  setDetailLoading: (isLoading) => {
    return {
      type: 'SET_DETAIL_LOADING',
      payload: isLoading,
    };
  },
  getFile: (idType, id, filename) => {
    return {
      type: 'GET_FILE',
      payload: {idType, id, filename},
    };
  },
  getArchive: (reportMessageId, filename) => {
    return {
      type: 'GET_ARCHIVE',
      payload: {reportMessageId, filename},
    };
  },
  getReport: (reportMessageId, filename) => {
    return {
      type: 'GET_REPORT',
      payload: {reportMessageId, filename},
    };
  },
  setQuery: (param) => {
    return {
      type: 'SET_QUERY',
      payload: param
    };
  },
  clearQuery: () => {
    return {
      type: 'CLEAR_QUERY',
    };
  },
  resolveMessage: (messageId) => {
    return {
      type: 'RESOLVE_MSG',
      payload: messageId
    };
  },
  setResolvedMessage: (resolvedMessage) => {
    return {
      type: 'SET_RESOLVE_MSG',
      payload: resolvedMessage
    };
  },
  clearResolvedMessage: () => {
    return {
      type: 'CLEAR_RESOLVE_MSG',
    };
  },
  // Actions for Submission Form
  setUploadItem: (item) => {
    return {
      type: 'SET_UPLOAD_ITEM',
      payload: item
    };
  },
  resetUploadItem: () => {
    return {
      type: 'RESET_UPLOAD_ITEM',
    };
  },
  createUploadItem: () => {
    return {
      type: 'CREATE_UPLOAD_ITEM',
    };
  },
};

const initialState = {
  tag: 'pmds',
  tab: null,
  page: 0,
  totalPages: null,
  size: 10,
  records: [],
  recordLoading: false,
  detail: {},
  detailLoading: false,
  query: {},
  resolvedMessage: null,
  uploadItem: {},
};

const pmdsReducer = (state = initialState, action) => {
  let reduce, pmdsType, idType;
  switch (action.type) {
    case 'SET_TAB':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          tab: action.payload,
          uploadItem : {},
          query: {},
        });
      };
      return reduce(state, action);

    case 'SET_PAGE':
      reduce = (state, action) => {
        const {size, query} = state;
        let page;
        if (action.payload === null || action.payload === undefined) {
          page = initialState.page;
        } else {
          page = action.payload;
        }
        pmdsType = getPMDSTypeFromTab(state.tab);
        getPMDSHistory(pmdsType, page, size, query);
        return Object.assign(Object.assign({}, state), {
          page: page,
        });
      };
      return reduce(state, action);

    case 'SET_SIZE':
      reduce = (state, action) => {
        const { query } = state;
        const newPage = 0;
        let newSize;
        if (action.payload === null || action.payload === undefined) {
          newSize = initialState.size;
        } else {
          newSize = action.payload;
        }
        pmdsType = getPMDSTypeFromTab(state.tab);
        getPMDSHistory(pmdsType, newPage, newSize, query);
        return Object.assign(Object.assign({}, state), {
          page: newPage,
          size: newSize,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_PAGES':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalPages: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalRecords: action.payload,
        });
      };
      return reduce(state, action)

    case 'GET_RECORDS':
      // TODO: handle query param
      reduce = (state, action) => {
        const {page, size, query} = state;
        pmdsType = getPMDSTypeFromTab(state.tab);
        getPMDSHistory(pmdsType, page, size, query);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          records: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_RECORD_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          recordLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_QUERY':
      reduce = (state, action) => {
        const queryItem = Object.assign({}, state.query);
        return Object.assign(Object.assign({}, state), {
          query: Object.assign(queryItem, action.payload),
        });
      };
      return reduce(state, action);

    case 'CLEAR_QUERY':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {query: {}, page: initialState.page});
      };
      return reduce(state, action);

    case 'GET_DETAIL':
      pmdsType = getPMDSTypeFromTab(state.tab);
      getPMDSDetail(pmdsType, action.payload.idType, action.payload.id);

      return store.getState();

    case 'SET_DETAIL':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detail: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_DETAIL_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detailLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'GET_FILE':
      pmdsType = getPMDSTypeFromTab(state.tab);
      downloadPMDSFile(pmdsType, action.payload.idType, action.payload.id, action.payload.filename);
      return store.getState();

    case 'GET_ARCHIVE':
      pmdsType = getPMDSTypeFromTab(state.tab);
      downloadPMDSArchive(pmdsType, action.payload.reportMessageId, action.payload.filename);
      return store.getState();

    case 'GET_REPORT':
      pmdsType = getPMDSTypeFromTab(state.tab);
      downloadPMDSReport(pmdsType, action.payload.reportMessageId, action.payload.filename);
      return store.getState();

    case 'RESOLVE_MSG':
      const {resolvedMessage} = state;
      pmdsType = getPMDSTypeFromTab(state.tab);
      idType = 'messageId';
      resolvePMDSMessage(pmdsType, idType, action.payload, resolvedMessage);
      return store.getState();

    case 'SET_RESOLVE_MSG':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          resolvedMessage: action.payload,
        });
      };
      return reduce(state, action);

    case 'CLEAR_RESOLVE_MSG':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {resolvedMessage: null});
      };
      return reduce(state, action);

    case 'SET_UPLOAD_ITEM':
      reduce = (state, action) => {
        const uploadItem = Object.assign({}, state.uploadItem);
        return Object.assign(Object.assign({}, state), {
          uploadItem: Object.assign(uploadItem, action.payload),
        });
      };
      return reduce(state, action);

    case 'RESET_UPLOAD_ITEM':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {uploadItem: {}});
      };
      return reduce(state, action);

    case 'CREATE_UPLOAD_ITEM':
      if (state.tab === 'submission') {
        submissionUpload(state.uploadItem);
      } else if (state.tab === 'enquiry') {
        enquiryUpload(state.uploadItem);
      }
      return store.getState();

    default:
      return state;
  }
};

export const store = createStore(pmdsReducer);

const getPMDSTypeFromTab = (tab) => {
  if (tab === 'subhistory') {
    return 'submission';
  } else if (tab === 'enqhistory') {
    return 'enquiry';
  }
};

const getPMDSHistory = (pmdsType, page, size, params = {}) => {
  const curTab = store.getState().tab;
  store.dispatch(actions.setRecordLoading(true));
  const url = convertUrlQueryParams(`${APIPrefix}/crp-service/pmds/${pmdsType}/history`, Object.assign({page, size}, params));
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    const {tab} = store.getState();
    if (tab !== curTab) {
      return;
    }
    if (result.result === "Success") {
      store.dispatch(actions.setRecords(result.data.record ? result.data.record : []));
      store.dispatch(actions.setTotalPages(result.data.totalPages));
      store.dispatch(actions.setTotalRecords(result.data.total));
      if (result.data && result.data.total === 0) {
        alert('No matched record found');
      }
    } else {
      displayActionResponse(result);
    }
    store.dispatch(actions.setRecordLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const submissionUpload = (submissionItem) => {
  const {dataFile, fileFormat, recordCount, messageRemark, attachments} = submissionItem;
  let body = new FormData();
  body.append('dataFile', dataFile);
  body.append('fileFormat', fileFormat);
  if (recordCount !== undefined && recordCount !== null && recordCount !== '') {
    body.append('recordCount', recordCount);
  }
  if (messageRemark !== undefined && messageRemark !== null && messageRemark !== '') {
    body.append('messageRemark', messageRemark);
  }
  if (attachments !== undefined && attachments !== null && attachments.length) {
    body.append('attachments', attachments);
  }
  fetch(`${APIPrefix}/crp-service/pmds/submission/upload`, {
    method: 'POST',
    headers: getJsonRequestHeaderWithCsrf(),
    body: body
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    if (result.result === "Success") {
      store.dispatch(actions.resetUploadItem());
      store.dispatch(actions.setTab('subhistory'));
    }
  }).catch(error => {
    console.error(error);
  });
};

const enquiryUpload = (enquiryItem) => {
  const {dataFile, fileFormat, messageRemark, departmentCode, targetedCRAs} = enquiryItem;
  let body = new FormData();
  body.append('dataFile', dataFile);
  body.append('fileFormat', fileFormat);
  body.append('targetedCRAs', targetedCRAs);
  if (messageRemark !== undefined && messageRemark !== null) {
    body.append('messageRemark', messageRemark);
  }
  if (departmentCode !== undefined && departmentCode !== null && departmentCode !== '') {
    body.append('recordCount', departmentCode);
  }
  fetch(`${APIPrefix}/crp-service/pmds/enquiry/upload`, {
    method: 'POST',
    headers: getJsonRequestHeaderWithCsrf(),
    body: body
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    if (result.result === "Success") {
      store.dispatch(actions.resetUploadItem());
      store.dispatch(actions.setTab('enqhistory'));
    }
  }).catch(error => {
    console.error(error);
  });
};

const getPMDSDetail = (pmdsType, idType, messageId) => {
  GetPMDSDetail(store, actions, pmdsType, idType, messageId);
};

export const GetPMDSDetail = (curStore, curActions, pmdsType, idType, messageId) => {
  curStore.dispatch(curActions.setDetailLoading(true));
  const url = `${APIPrefix}/crp-service/pmds/${pmdsType}/detail/${idType}/${messageId}`;
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      curStore.dispatch(curActions.setDetail(result.data));
    } else {
      displayActionResponse(result);
    }
    curStore.dispatch(curActions.setDetailLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const resolvePMDSMessage = (pmdsType, idType, messageId, resolvedMessage) => {
  const body = {resolvedMessage};
  fetch(`${APIPrefix}/crp-service/pmds/${pmdsType}/detail/${idType}/${messageId}`, {
    method: 'PUT',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    if (result.result === "Success") {
      store.dispatch(actions.clearResolvedMessage());
      store.dispatch(actions.getDetail(idType, messageId));
      store.dispatch(actions.getRecords());
    }
  }).catch(error => {
    console.error(error);
  });
};

const downloadPMDSFile = (pmdsType, idType, messageId, filename) => {
  const url = `${APIPrefix}/crp-service/pmds/${pmdsType}/detail/${idType}/${messageId}/file`;
  fetch(url).then(response => {
    handleDownloadResponse(response, filename);
  }).catch(error => {
    console.error(error);
  });

};

const downloadPMDSArchive = (pmdsType, reportMessageId, filename) => {
  const url = `${APIPrefix}/crp-service/pmds/${pmdsType}/detail/messageId/${reportMessageId}/report`;
  fetch(url).then(response => {
    handleDownloadResponse(response, filename, 'zip');
  }).catch(error => {
    console.error(error);
  });

};

const downloadPMDSReport = (pmdsType, reportMessageId, filename) => {
  const url = `${APIPrefix}/crp-service/pmds/${pmdsType}/detail/messageId/${reportMessageId}/report/${filename}`;
  fetch(url).then(response => {
    handleDownloadResponse(response, filename);
  }).catch(error => {
    console.error(error);
  });

};
