"use strict";

import {createStore, buildTree, ErrorMessage, displayActionResponse} from '../common.js';
import {APIPrefix, getJsonRequestHeaderWithCsrf} from '../common.js';

// Redux architecture pieces
export const actions = {
  initItems: () => {
    return {
      type: 'INIT_ITEMS'
    };
  },
  getItems: () => {
    // Get items from fetch
    return {
      type: 'GET_ITEMS'
    };
  },
  setItems: (items) => {
    // To be called by getItems callback
    return {
      type: 'SET_ITEMS',
      payload: items,
    };
  },
  setPage: (page = 0) => {
    // Set currentPage & currentPageItems
    return {
      type: 'SET_PAGE',
      payload: page,
    };
  },
  setQueryParams: (params) => {
    // Set Query params for getting items
    // Make callback to getItems
    return {
      type: 'SET_QUERY',
      payload: params,
    };
  },
  clearQueryParams: () => {
    return {
      type: 'CLEAR_QUERY',
    };
  },
  getFlattenRoleTree: () => {
    return {
      type: 'GET_FLATTEN_ROLETREE',
    };
  },
  setRoleTree: (data) => {
    // Set data for rendering roletree
    return {
      type: 'SET_ROLETREE',
      payload: data,
    };
  },
  initRoleStates: (id) => {
    return {
      type: 'INIT_ROLE_STATES',
      payload: id,
    };
  },
  getRoleStates: (id, state) => {
    return {
      type: 'GET_ROLE_STATES',
      payload: {
        id,
        state,
      },
    };
  },
  setChangeItem: (item) => {
    // Set modified item values to store
    // To be submitted to backend
    return {
      type: 'SET_CHANGE_ITEM',
      payload: item,
    };
  },
  clearChangeItem: () => {
    return {
      type: 'CLEAR_CHANGE_ITEM',
    };
  },
  updateItem: (item) => {
    return {
      type: 'UPDATE_ITEM',
    };
  },
  deleteItem: () => {
    return {
      type: 'DELETE_ITEM',
    };
  },
  createItem: () => {
    return {
      type: 'CREATE_ITEM',
    };
  },
};

const initialState = {
  queryParams: {},
  items: [],
  currentPage: 1,
  itemPerPage: 10,
  currentPageItems: [],
  changeItem: {},
  roleTree: null,
  roleStates: [],
};


const userGroupReducer = (state = initialState, action) => {
  let permissionIds, item;
  switch (action.type) {
    case 'INIT_ITEMS':
      initGroupItems();
      return state;

    case 'GET_ITEMS':
      getGroupItems(state.queryParams);
      return state;

    case 'SET_ITEMS':
      // Set items & initialize page + currentPageItems
      return Object.assign(Object.assign({}, state), {
        items: action.payload,
        currentPage: 1,
        currentPageItems: action.payload.slice(0, state.itemPerPage)
      });

    case 'SET_PAGE':
      const page = action.payload;
      const sIdx = (page - 1) * state.itemPerPage;
      const eIdx = (page) * state.itemPerPage;
      return Object.assign(Object.assign({}, state), {
        currentPage: page,
        currentPageItems: state.items.slice(sIdx, eIdx),
      });

    case 'SET_QUERY':
      let queryParams = Object.assign(
         Object.assign({}, state.queryParams), action.payload);
      return Object.assign(Object.assign({}, state), {queryParams});

    case 'CLEAR_QUERY':
      return Object.assign(Object.assign({}, state), {queryParams: initialState.queryParams, page: 0});

    case 'GET_FLATTEN_ROLETREE':
      getFlattenRoleTree();
      return state;

    case 'SET_ROLETREE':
      const flattenData = convertRawRoleTreeData(action.payload.nodes, action.payload.roles);
      return Object.assign(Object.assign({}, state), {
        roleTree: makeRoleTree(flattenData),
      });

    case 'INIT_ROLE_STATES':
      item = state.items.find(itm => itm.id === action.payload);
      return Object.assign(Object.assign({}, state), {
        roleStates: initRoleStates(
          state.roleTree,
          item && item.permissionIds ? item.permissionIds : []
        ),
      });

    case 'GET_ROLE_STATES':
      return Object.assign(Object.assign({}, state), {
        roleStates: getNewRoleStates(
          state.roleTree,
          state.roleStates,
          action.payload
        ),
      });

    case 'SET_CHANGE_ITEM':
      let changeItem = Object.assign(
         Object.assign({}, state.changeItem), action.payload);
      return Object.assign(Object.assign({}, state), {changeItem});

    case 'CLEAR_CHANGE_ITEM':
      return Object.assign(Object.assign({}, state), {changeItem: {}});

    case 'UPDATE_ITEM':
      permissionIds = roleStatesToPermissionIds(state.roleTree, state.roleStates);
      item = state.items.find(itm => itm.id === state.changeItem.id);
      updateUserGroup(Object.assign(Object.assign({}, item), state.changeItem), permissionIds);
      return state;

    case 'DELETE_ITEM':
      deleteUserGroup(state.changeItem.id);
      return state;

    case 'CREATE_ITEM':
      permissionIds = roleStatesToPermissionIds(state.roleTree, state.roleStates);
      postUserGroup(state.changeItem, permissionIds);
      return state;

    default:
      return state;
  }
};

export const store = createStore(userGroupReducer);

const initGroupItems = () => {
  fetch(`${APIPrefix}/group`).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      return result.data;
    } else {
      displayActionResponse(result);
    }
  }).then(data => {
    store.dispatch(actions.setItems(data));
    store.dispatch(actions.getFlattenRoleTree());
  }).catch(error => {
    console.error(error);
  });
};

const getGroupItems = (params) => {
  let url = `${APIPrefix}/group`;
  const keys = Object.keys(params);
  if (keys.length) {
    keys.forEach((k, idx) => {
      if (params[k] !== '') {
        if (idx === 0) {
          url = `${url}?`;
        } else {
          url = `${url}&`;
        }
        url = `${url}${k}=${params[k]}`
      }
    });
  }
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      store.dispatch(actions.setItems(result.data));
    } else {
      displayActionResponse(result);
    }
  }).catch(error => {
    console.error(error);
  });
};

const getFlattenRoleTree = () => {
  GetFlattenRoleTree(store, actions);
};

export const GetFlattenRoleTree = (curStore, curActions) => {
  fetch(`${APIPrefix}/role/tree/flatten`).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      return result.data;
    } else {
      displayActionResponse(result);
    }
  }).then(data => {
    curStore.dispatch(curActions.setRoleTree(data));
  }).catch(error => {
    console.error(error);
    // alert("Failed to get Flatten Role Tree from server");
  });
};

const postUserGroup = (changeItem, permissionIds) => {
  const body = Object.assign({}, changeItem);
  body.permissionIds = permissionIds;
  fetch(`${APIPrefix}/group`, {
    method: 'POST',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const deleteUserGroup = (id) => {
  fetch(`${APIPrefix}/group/${id}`, {
    method: 'DELETE',
    headers: getJsonRequestHeaderWithCsrf(),
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const updateUserGroup = (body, permissionIds) => {
  const id = body.id;
  body.permissionIds = permissionIds;
  fetch(`${APIPrefix}/group/${id}`, {
    method: 'PUT',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

export const convertRawRoleTreeData = (nodes, roles) => {
  let ret = [];
  nodes.forEach(n => {
    ret.push({
      id: `N${n.id}`,
      pid: n.parentNodeId ? `N${n.parentNodeId}` : null,
      name: n.name,
      isRole: false,
    });
  });
  roles.forEach(r => {
    ret.push({
      id: r.id,
      pid: r.nodeId ? `N${r.nodeId}` : null,
      requiredId: r.requiredId,
      name: r.name,
      isRole: true,
    });
  });
  return ret;
};

export const makeRoleTree = (data) => {
  const tree = buildTree();
  const rootId = -999;
  tree.setRoot(rootId, {
    name: 'Portal',
    isRole: false,
  });
  if (data.length > 0) {
    data.forEach(d => {
      let parent;
      const pid = d.pid === null ? rootId : d.pid;
      if (tree.contains(pid)) {
        parent = tree.getNodeById(pid);
      } else {
        parent = tree.addNew(pid, {});
      }
      tree.addChildTo(parent, d.id, {
        name: d.name,
        isRole: d.isRole,
        requiredId: d.requiredId ? d.requiredId : null,
      });
    });
    // Delete leaf that not isRole
    const deleteNodes = tree.preOrder().filter(n => tree.isLeaf(n) && !n.getProperty('isRole'));
    deleteNodes.forEach(n => {
      tree.deleteLeaf(n);
    });
  }

  // Set state index
  tree.preOrder().forEach((n, idx) => {
    n.setProperty('stateIndex', idx);
  });
  return tree;
};

export const initRoleStates = (tree, permissionIds) => {
  const roleStates = [];
  tree.preOrder().forEach(n => {
    if (tree.isLeaf(n)) {
      if (permissionIds.findIndex(id => id === n.id) >= 0) {
        roleStates.push(1);
      } else {
        roleStates.push(0);
      }
    } else {
      // Just fill some value first
      roleStates.push(2);

    }
  });
  const pNodes = tree.preOrder().filter(n => !tree.isLeaf(n));
  pNodes.forEach(pNode => {
    let allChecked = true;
    let allUnchecked = true;
    const leafs = tree.preOrder(pNode).filter(n => tree.isLeaf(n));
    leafs.forEach(n => {
      const roleState = roleStates[n.getProperty('stateIndex')];
      allChecked = allChecked && (roleState === 1);
      allUnchecked = allUnchecked && (roleState === 0);
    });
    const pNodeStateIndex = pNode.getProperty('stateIndex');
    if (allChecked) {
      roleStates[pNodeStateIndex] = 1;
    } else if (allUnchecked) {
      roleStates[pNodeStateIndex] = 0;
    } else {
      roleStates[pNodeStateIndex] = 2;
    }
  });
  return roleStates;
};

export const getNewRoleStates = (tree, roleStates, payload) => {
  const node = tree.getNodeById(payload.id);
  const isRole = node.getProperty('isRole');
  let newRoleStates;
  let updatedLeafIds = [];
  if (roleStates.length) {
    newRoleStates = Array.from(roleStates);
  } else {
    newRoleStates = tree.preOrder().map(n => 0);
  }
  newRoleStates[node.getProperty('stateIndex')] = payload.state;
  // triggered node is not leaf, update all child
  if (!isRole) {
    if (payload.state === 0 || payload.state === 1) {
      tree.preOrder(node).forEach(n => {
        let oldState = newRoleStates[n.getProperty('stateIndex')];
        if (oldState !== payload.state && tree.isLeaf(n)) {
          updatedLeafIds.push(n.id);
        }
        newRoleStates[n.getProperty('stateIndex')] = payload.state;
      });
    }
  } else {
    updatedLeafIds.push(node.id);
  }

  // check requiredId and update ancestor for all updated leaf
  const leaves = tree.preOrder().filter(n => tree.isLeaf(n));
  let nextLeafIds = [];
  updatedLeafIds = Array.from(new Set(updatedLeafIds));
  updatedLeafIds.forEach(id => {
    const leaf = tree.getNodeById(id);
    let ids = updateRoleStatesByRequiredId(newRoleStates, leaves, leaf, newRoleStates[leaf.getProperty('stateIndex')]);
    nextLeafIds = nextLeafIds.concat(ids);
    updateAncestorRoleStates(newRoleStates, tree, leaf);
  });

  nextLeafIds = Array.from(new Set(nextLeafIds));
  nextLeafIds.filter(id => updatedLeafIds.indexOf(id) < 0);
  nextLeafIds.forEach(id => {
    const leaf = tree.getNodeById(id);
    updateAncestorRoleStates(newRoleStates, tree, leaf);
  });

  return newRoleStates;
};

const updateAncestorRoleStates = (newRoleStates, tree, node) => {
  const pNodes = tree.getAncestors(node);
  pNodes.forEach(pNode => {
    let allChecked = true;
    let allUnchecked = true;
    const leafs = tree.preOrder(pNode).filter(n => tree.isLeaf(n));
    leafs.forEach(n => {
      const roleState = newRoleStates[n.getProperty('stateIndex')];
      allChecked = allChecked && (roleState === 1);
      allUnchecked = allUnchecked && (roleState === 0);
    });
    const pNodeStateIndex = pNode.getProperty('stateIndex');
    if (allChecked) {
      newRoleStates[pNodeStateIndex] = 1;
    } else if (allUnchecked) {
      newRoleStates[pNodeStateIndex] = 0;
    } else {
      newRoleStates[pNodeStateIndex] = 2;
    }
  });
};

const updateRoleStatesByRequiredId = (newRoleStates, leaves, node, state) => {
  const requiredId = node.getProperty('requiredId');
  const updatedNodeIds = [];
  if (requiredId !== null && state === 1) {
    const findRequiredNode = (rid) => leaves.find(n => n.id === rid);
    let curRequiredId = requiredId;
    while (curRequiredId !== null) {
      const requiredNode = findRequiredNode(curRequiredId);
      newRoleStates[requiredNode.getProperty('stateIndex')] = 1;
      updatedNodeIds.push(requiredNode.id);
      curRequiredId = requiredNode.getProperty('requiredId');
    }
  }
  if (state === 0) {
    const findDependentNode = (id) => leaves.filter(n => n.getProperty('requiredId') === id);
    let dependentNodes = findDependentNode(node.id);
    while (dependentNodes.length > 0) {
      const newDependentNodes = [];
      dependentNodes.forEach(n => {
        newRoleStates[n.getProperty('stateIndex')] = 0;
        updatedNodeIds.push(n.id);
        findDependentNode(n.id).forEach(dn => {
          newDependentNodes.push(dn);
        });
      });
      dependentNodes = newDependentNodes;
    }
  }
  return updatedNodeIds;
}

export const roleStatesToPermissionIds = (tree, roleStates) => {
  const ret = [];
  tree.preOrder().filter(n => tree.isLeaf(n)).forEach(n => {
    if (roleStates[n.getProperty('stateIndex')] === 1) {
      ret.push(n.id);
    }
  });
  return ret;
};

export const getPermissionIdsToRoleNameMap = (tree) => {
  const ret = {};
  tree.preOrder().filter(n => tree.isLeaf(n)).forEach(n => {
    ret[n.id] = n.getProperty("name");
  });
  return ret;
};
