"use strict";

import {createStore, ErrorMessage, displayActionResponse} from '../common.js';
import {APIPrefix, getJsonRequestHeaderWithCsrf} from '../common.js';
import {camelCaseToSentenceCase} from '../common.js';
import {exportFullRecordCSV, getReorderedHeadersAndRecArrs} from '../component/history_export.js';

// Redux architecture pieces
export const actions = {
  getCategories: () => {
    return {
      type: 'GET_CATEGORIES',
    };
  },
  setCategories: (categories) => {
    return {
      type: 'SET_CATEGORIES',
      payload: categories,
    };
  },
  setConfigItems: (configItems) => {
    return {
      type: 'SET_CONFIG_ITEMS',
      payload: configItems,
    };
  },
  getConfigItems: (category) => {
    return {
      type: 'GET_CONFIG_ITEMS',
      payload: category,
    };
  },
  setChangeItems: (changeItems) => {
    return {
      type: 'SET_CHANGE_ITEMS',
      payload: changeItems,
    };
  },
  postChangeItems: () => {
    return {
      type: 'POST_CHANGE_ITEMS',
    };
  },
  setExportLoading: (exportLoading) => {
    return {
      type: 'SET_EXPORT_LOADING',
      payload: exportLoading,
    };
  },
  exportConfigItems: () => {
    return {
      type: 'EXPORT_CONFIG_ITEMS',
    };
  },
};

const initialState = {
  category: null,
  categories: [],
  configItems: [],
  changeItems: [],
  exportLoading: false,
};

// Constant
const gatewayConfigReducer = (state = initialState, action) => {
  switch (action.type) {
    case 'GET_CATEGORIES':
      getGatewayConfigCategories();
      return state;

    case 'SET_CATEGORIES':
      const category = action.payload.length ? action.payload[0].category : null;
      return {
        category: category,
        categories: action.payload,
        configItems: [],
        changeItems: [],
        exportLoading: state.exportLoading,
      };

    case 'SET_CONFIG_ITEMS':
      return {
        category: state.category,
        categories: state.categories,
        configItems: action.payload,
        changeItems: state.changeItems,
        exportLoading: state.exportLoading,
      };

    case 'GET_CONFIG_ITEMS':
      getGatewayConfigItems(action.payload);
      return {
        category: action.payload,
        categories: state.categories,
        configItems: state.configItems,
        changeItems: state.changeItems,
        exportLoading: state.exportLoading,
      };

    case 'SET_CHANGE_ITEMS':
      return {
        category: state.category,
        categories: state.categories,
        configItems: state.configItems,
        changeItems: action.payload,
        exportLoading: state.exportLoading,
      };

    case 'POST_CHANGE_ITEMS':
      updateGatewayConfigItems(state.category, state.changeItems);
      return {
        category: state.category,
        categories: state.categories,
        configItems: [],
        changeItems: [],
        exportLoading: state.exportLoading,
      };

    case 'SET_CHANGE_ITEMS':
      return {
        category: state.category,
        categories: state.categories,
        configItems: state.configItems,
        changeItems: action.payload,
        exportLoading: state.exportLoading,
      };

    case 'SET_EXPORT_LOADING':
      return {
        category: state.category,
        categories: state.categories,
        configItems: state.configItems,
        changeItems: state.changeItems,
        exportLoading: action.payload,
      };

    case 'EXPORT_CONFIG_ITEMS':
      exportFullGatewayConfigItems();
      return state;

    default:
      return state;
  }
};

export const store = createStore(gatewayConfigReducer);

const getGatewayConfigCategories = () => {
  fetch(`${APIPrefix}/configuration/category`).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      const results = result.data;
      return result.data;
    }
  }).then(data => {
    store.dispatch(actions.setCategories(data));
  }).catch(error => {
    console.error(error);
    alert("Failed to get Gateway Config Categories from server");
  });
};

const getGatewayConfigItems = (category) => {
  fetch(`${APIPrefix}/configuration/${category}`).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      const results = result.data;
      return result.data;
    }
  }).then(data => {
    store.dispatch(actions.setConfigItems(data));
  }).catch(error => {
    console.error(error);
    alert("Failed to get Gateway Config Items from server");
  });
};

const updateGatewayConfigItems = (category, changeItems) => {
  fetch(`${APIPrefix}/configuration/${category}`, {
    method: 'POST',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify({changeItems})
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result) {
      displayActionResponse(result);
      if (result.result === "Success") {
      }
    }
  }).then(() => {
    getGatewayConfigItems(category);
  }).catch(error => {
    console.error(error);
    alert(`Error: ${error.message || 'Failed to update gateway configuration'}`);
    // Reload config items on error to maintain state
    getGatewayConfigItems(category);
  });
};

const exportFullGatewayConfigItems = () => {
  const prepareExportData = (data) => {
    // Map status / category value
    const {categories} = store.getState();
    const statusMap = [
      {
        key: 0,
        value: "Inactive",
      },
      {
        key: 1,
        value: "Active",
      },
    ];
    data.forEach(d => {
      d.categoryName = d.category === 0 ? 'Custom' : categories.find(c => c.category === d.category).categoryName;
      delete d.category;
      d.status = statusMap.find(s => s.key === d.status).value;
    });
    // Reorder column
    const order = ['id', 'categoryName', 'key', 'displayName', 'value', 'status', 'validUntil'];
    let {headers, recArrs} = getReorderedHeadersAndRecArrs(data, order);
    headers = headers.map(camelCaseToSentenceCase);
    return {headers, recArrs};
  };
  store.dispatch(actions.setExportLoading(true));
  fetch(`${APIPrefix}/configuration`).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      return prepareExportData(result.data);
    }
    displayActionResponse(result);
  }).then(exportData => {
    exportFullRecordCSV('gateway_config_export.csv', exportData.headers, exportData.recArrs);
    store.dispatch(actions.setExportLoading(false));
  }).catch(error => {
    console.error(error);
    alert("Failed to export All Categories Gateway Configurations");
    store.dispatch(actions.setExportLoading(false));
  });
}
