"use strict";

import {createStore, displayActionResponse} from '../common.js';
import {APIPrefix, getJsonRequestHeaderWithCsrf} from '../common.js';

// Redux architecture pieces
export const actions = {
  initItems: () => {
    return {
      type: 'INIT_ITEMS'
    };
  },
  setInitItems: (items) => {
    // To be called by getItems callback
    return {
      type: 'SET_INIT_ITEMS',
      payload: items,
    };
  },
  getItems: () => {
    // Get items from fetch
    return {
      type: 'GET_ITEMS'
    };
  },
  setItems: (items) => {
    // To be called by getItems callback
    return {
      type: 'SET_ITEMS',
      payload: items,
    };
  },
  setPage: (page = 0) => {
    // Set currentPage & currentPageItems
    return {
      type: 'SET_PAGE',
      payload: page,
    };
  },
  setQueryParams: (params) => {
    // Set Query params for getting items
    // Make callback to getItems
    return {
      type: 'SET_QUERY',
      payload: params,
    };
  },
  clearQueryParams: () => {
    return {
      type: 'CLEAR_QUERY',
    };
  },
  setChangeItem: (item) => {
    // Set modified item values to store
    // To be submitted to backend
    return {
      type: 'SET_CHANGE_ITEM',
      payload: item,
    };
  },
  clearChangeItem: () => {
    return {
      type: 'CLEAR_CHANGE_ITEM',
    };
  },
  updateItem: (item) => {
    return {
      type: 'UPDATE_ITEM',
    };
  },
  deleteItem: () => {
    return {
      type: 'DELETE_ITEM',
    };
  },
  createItem: () => {
    return {
      type: 'CREATE_ITEM',
    };
  },
};

const initialState = {
  queryParams: {},
  items: [],
  currentPage: 1,
  itemPerPage: 10,
  currentPageItems: [],
  changeItem: {},
};


const departmentReducer = (state = initialState, action) => {
  let changeItem;
  switch (action.type) {
    case 'INIT_ITEMS':
      initDepartmentItems();
      return state;

    case 'SET_INIT_ITEMS':
      // For triggering create layout
      return Object.assign(Object.assign({}, state), {
        items: action.payload,
        currentPage: 1,
        currentPageItems: action.payload.slice(0, state.itemPerPage)
      });

    case 'GET_ITEMS':
      getDepartmentItems(state.queryParams);
      return state;

    case 'SET_ITEMS':
      // Set items & initialize page + currentPageItems
      return Object.assign(Object.assign({}, state), {
        items: action.payload,
        currentPage: 1,
        currentPageItems: action.payload.slice(0, state.itemPerPage)
      });

    case 'SET_PAGE':
      const page = action.payload;
      const sIdx = (page - 1) * state.itemPerPage;
      const eIdx = (page) * state.itemPerPage ;
      return Object.assign(Object.assign({}, state), {
        currentPage: page,
        currentPageItems: state.items.slice(sIdx, eIdx),
      });

    case 'SET_QUERY':
      let queryParams = Object.assign(
         Object.assign({}, state.queryParams), action.payload);
      return Object.assign(Object.assign({}, state), {queryParams});

    case 'CLEAR_QUERY':
      return Object.assign(Object.assign({}, state), {queryParams: initialState.queryParams, page: 0});

    case 'SET_CHANGE_ITEM':
      changeItem = Object.assign(
         Object.assign({}, state.changeItem), action.payload);
      return Object.assign(Object.assign({}, state), {changeItem});

    case 'CLEAR_CHANGE_ITEM':
      return Object.assign(Object.assign({}, state), {changeItem: {}});

    case 'UPDATE_ITEM':
      let item = state.items.find(itm => itm.id === state.changeItem.id);
      updateDepartment(Object.assign(Object.assign({}, item), state.changeItem));
      return state;

    case 'DELETE_ITEM':
      deleteDepartment(state.changeItem.departmentCode);
      return state;

    case 'CREATE_ITEM':
      postDepartment(state.changeItem);
      return state;

    default:
      return state;
  }
};

export const store = createStore(departmentReducer);

const initDepartmentItems = () => {
  fetch(`${APIPrefix}/department`).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      store.dispatch(actions.setInitItems(result.data));
    } else {
      const resultStatus = result && result.result ? result.result : 'Error:';
      const errMsg = result && result.resp_msg ?
      result.resp_msg : (result && result.error ? result.error : '');
      alert(`${resultStatus} ${errMsg}`);
    }
  }).catch(error => {
    console.error(error);
  });
};

const getDepartmentItems = (params) => {
  let url = `${APIPrefix}/department`;
  const keys = Object.keys(params);
  if (keys.length) {
    keys.filter(k => params[k] && params[k] !== "").forEach((k, idx) => {
      if (idx === 0) {
        url = `${url}?`;
      } else {
        url = `${url}&`;
      }
      url = `${url}${k}=${params[k]}`
    });
  }
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      store.dispatch(actions.setItems(result.data));
    } else {
      displayActionResponse(result);
    }
  }).catch(error => {
    console.error(error);
  });
};

const postDepartment = (changeItem) => {
  const body = Object.assign({}, changeItem);
  fetch(`${APIPrefix}/department`, {
    method: 'POST',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const deleteDepartment = (departmentCode) => {
  fetch(`${APIPrefix}/department/${departmentCode}`, {
    method: 'DELETE',
    headers: getJsonRequestHeaderWithCsrf(),
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const updateDepartment = (body) => {
  const departmentCode = body.departmentCode;
  fetch(`${APIPrefix}/department/${departmentCode}`, {
    method: 'PUT',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};
