"use strict";

import {createStore, displayActionResponse} from '../common.js';
import {APIPrefix, getRequestHeaderWithCsrf, convertUrlQueryParams} from '../common.js';
import {handleDownloadResponse} from '../common.js';
import {GetDataSubmissionDetail} from './datasub_model.js';
import {GetCreditReportDetail} from './credrpt_model.js';
import {GetPMDSDetail} from './pmds_model.js';
import {GetBusinessDetail} from './business_model.js';
import {GetFlattenRoleTree, convertRawRoleTreeData, makeRoleTree, initRoleStates} from './group_model.js';

// Redux architecture pieces
export const actions = {
  setTab: (tab) => {
    return {
      type: 'SET_TAB',
      payload: tab,
    };
  },
  setPage: (page) => {
    return {
      type: 'SET_PAGE',
      payload: page
    };
  },
  setSize: (size) => {
    return {
      type: 'SET_SIZE',
      payload: size
    };
  },
  setTotalPages: (totalPages) => {
    return {
      type: 'SET_TOTAL_PAGES',
      payload: totalPages
    };
  },
  setTotalRecords: (totalRecords) => {
    return {
      type: 'SET_TOTAL_RECORDS',
      payload: totalRecords
    };
  },
  getRecords: () => {
    return {
      type: 'GET_RECORDS',
    };
  },
  setRecords: (records) => {
    return {
      type: 'SET_RECORDS',
      payload: records,
    };
  },
  setRecordLoading: (isLoading) => {
    return {
      type: 'SET_RECORD_LOADING',
      payload: isLoading,
    };
  },
  getDetail: (id) => {
    return {
      type: 'GET_DETAIL',
      payload: id,
    };
  },
  setDetail: (detail) => {
    return {
      type: 'SET_DETAIL',
      payload: detail,
    };
  },
  getDetailInfo: (id) => {
    return {
      type: 'GET_DETAIL_INFO',
      payload: id,
    };
  },
  setDetailInfo: (detailId, requestType, detailUser) => {
    return {
      type: 'SET_DETAIL_INFO',
      payload: {detailId, requestType, detailUser},
    };
  },
  unsetDetailInfo: () => {
    return {
      type: 'UNSET_DETAIL_INFO',
    };
  },
  setDetailLoading: (isLoading) => {
    return {
      type: 'SET_DETAIL_LOADING',
      payload: isLoading,
    };
  },
  setQuery: (param) => {
    return {
      type: 'SET_QUERY',
      payload: param
    };
  },
  clearQuery: () => {
    return {
      type: 'CLEAR_QUERY',
    };
  },
  actionRequest: (id, action) => {
    return {
      type: 'ACTION_REQUEST',
      payload: {id, action},
    };
  },
  getFlattenRoleTree: () => {
    return {
      type: 'GET_FLATTEN_ROLETREE',
    };
  },
  setRoleTree: (data) => {
    // Set data for rendering roletree
    return {
      type: 'SET_ROLETREE',
      payload: data,
    };
  },
  getFile: (params) => {
    return {
      type: 'GET_FILE',
      payload: params,
    };
  },
};

const initialState = {
  tag: 'approval',
  tab: null,
  page: 0,
  totalPages: null,
  size: 10,
  records: [],
  recordLoading: false,
  detail: {},
  requestType: null,
  lastRequestType: null,
  detailUser: null,
  detailId: null,
  detailLoading: false,
  query: {},
  roleTree: null,
  userGroups: [],
  departments: [],
};

const approvalReducer = (state = initialState, action) => {
  let reduce, idType;
  switch (action.type) {
    case 'SET_TAB':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          tab: action.payload,
          records: initialState.records,
          query: initialState.query,
          page: initialState.page,
          totalPages: initialState.totalPages,
        });
      };
      return reduce(state, action);

    case 'SET_PAGE':
      reduce = (state, action) => {
        const {tab, size, query} = state;
        let page;
        if (action.payload === null || action.payload === undefined) {
          page = initialState.page;
        } else {
          page = action.payload;
        }
        getApprovalRecord(tab, page, size, query);
        return Object.assign(Object.assign({}, state), {
          page: page,
        });
      };
      return reduce(state, action);

    case 'SET_SIZE':
      reduce = (state, action) => {
        const { tab, query } = state;
        const newPage = 0;
        let newSize;
        if (action.payload === null || action.payload === undefined) {
          newSize = initialState.size;
        } else {
          newSize = action.payload;
        }
        getApprovalRecord(tab, newPage, newSize, query);
        return Object.assign(Object.assign({}, state), {
          page: newPage,
          size: newSize,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_PAGES':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalPages: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalRecords: action.payload,
        });
      };
      return reduce(state, action);

    case 'GET_RECORDS':
      // TODO: handle query param
      reduce = (state, action) => {
        const {tab, page, size, query} = state;
        getApprovalRecord(tab, page, size, query);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          records: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_RECORD_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          recordLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_QUERY':
      reduce = (state, action) => {
        const queryItem = Object.assign({}, state.query);
        return Object.assign(Object.assign({}, state), {
          query: Object.assign(queryItem, action.payload),
        });
      };
      return reduce(state, action);

    case 'CLEAR_QUERY':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {query: {}, page: initialState.page});
      };
      return reduce(state, action);

    case 'GET_DETAIL':
      reduce = (state, action) => {
        const {records} = state;
        getApprovalDetail(records, action.payload);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_DETAIL':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detail: action.payload,
        });
      };
      return reduce(state, action);

    case 'GET_DETAIL_INFO':
      reduce = (state, action) => {
        const {records} = state;
        getDetailInfo(records, action.payload);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_DETAIL_INFO':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          requestType: action.payload.requestType,
          detailUser: action.payload.detailUser,
          detailId: action.payload.detailId,
        });
      };
      return reduce(state, action);

    case 'UNSET_DETAIL_INFO':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          lastRequestType: state.requestType,
          requestType: initialState.requestType,
          detailUser: initialState.detailUser,
          detailId: initialState.detailId,
        });
      };
      return reduce(state, action);

    case 'SET_DETAIL_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detailLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'ACTION_REQUEST':
      approveRequest(action.payload.id, action.payload.action);
      return store.getState();

    case 'GET_FLATTEN_ROLETREE':
      getFlattenRoleTree();
      return store.getState();

    case 'SET_ROLETREE':
      reduce = (state, action) => {
        const {detailId} = state;
        const flattenData = convertRawRoleTreeData(action.payload.nodes, action.payload.roles);
        store.dispatch(actions.getDetail(detailId));
        return Object.assign(Object.assign({}, state), {
          roleTree: makeRoleTree(flattenData),
        });
      };
      return reduce(state, action);

    case 'GET_FILE':
      reduce = (state, action) => {
        const params = action.payload;
        downloadApprovalFile(params.requestType, params.id, params.filename);
        return store.getState();
      }
      return reduce(state, action);

    default:
      return state;
  }
};

export const store = createStore(approvalReducer);

const getApprovalRecord = (curTab, page, size, params = {}) => {
  store.dispatch(actions.setRecordLoading(true));
  const url = convertUrlQueryParams(`${APIPrefix}/approval/${curTab}`, Object.assign({page, size}, params));
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    const {tab} = store.getState();
    if (tab !== curTab) {
      return;
    }
    if (result.result === "Success") {
      store.dispatch(actions.setRecords(result.data.record ? result.data.record : []));
      store.dispatch(actions.setTotalPages(result.data.totalPages));
      store.dispatch(actions.setTotalRecords(result.data.total));
      if (result.data && result.data.total === 0) {
        alert('No matched record found');
      }
    } else {
      displayActionResponse(result);
    }
    store.dispatch(actions.setRecordLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const approveRequest = (id, action) => {
  if (['approve', 'reject', 'cancel'].findIndex(a => a === action) === -1) {
    console.error('Unregconized approve request action');
  }
  fetch(`${APIPrefix}/approval/${id}/${action}`, {
    method: 'POST',
    headers: getRequestHeaderWithCsrf(),
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    if (result.result === "Success") {
      store.dispatch(actions.unsetDetailInfo());
      store.dispatch(actions.getRecords());
    }
  }).catch(error => {
    console.error(error);
  });

};

const getDetailInfo = (records, id) => {
  const getInfo = () => {
    let {requestType, requestUser} = records.find(rec => rec.id === id);
    store.dispatch(actions.setDetailInfo(id, requestType, requestUser));
    if (requestType === 'DSUL' || requestType === 'DAUL' || requestType === 'IDUL' ||
        requestType === 'CRUL' || requestType === 'DCRR' ||
        requestType === 'PDUL' || requestType === 'PCUL' ||
        requestType === 'SDPC' || requestType === 'SDPU' || requestType === 'SDPD' ||
        requestType === 'SUSC' || requestType === 'SUSU' || requestType === 'SUSD' || requestType === 'SUSS' ||
        requestType === 'PPCR' || requestType === 'PWCR' ) {
      store.dispatch(actions.getDetail(id));
    // SUSC : 'Create User',
    // SUSU : 'Update User',
    } else if (requestType === 'SUGC' || requestType === 'SUGU' || requestType === 'SUGD') {
      store.dispatch(actions.getFlattenRoleTree());
    } else {
      console.error('requestType not regconized');
    }
  };
  setTimeout(getInfo, 1);
}

const getApprovalDetail = (records, id) => {
  let {requestType, requestUser, referenceData} = records.find(rec => rec.id === id);
  const urId = referenceData && referenceData.length && referenceData[0].uploadReferenceId ?  referenceData[0].uploadReferenceId : null;
  if (requestType === 'DSUL' || requestType === 'DAUL' || requestType === 'IDUL') {
    if (urId !== null && urId !== undefined) {
      getDXULDetail(urId);
    }
  } else if (requestType === 'CRUL') {
    if (urId !== null && urId !== undefined) {
      getCRULDetail(urId);
    }
  } else if (requestType === 'PDUL') {
    if (urId !== null && urId !== undefined) {
      getPDULDetail(urId);
    }
  } else if (requestType === 'PCUL') {
    if (urId !== null && urId !== undefined) {
      getPCULDetail(urId);
    }
  } else if (requestType === 'PPCR') {
    const mId = referenceData && referenceData.length && referenceData[0].messageId ? referenceData[0].messageId : null;
    getPPCRDetail(mId);
  } else if (requestType === 'PWCR') {
    const mId = referenceData && referenceData.length && referenceData[0].messageId ? referenceData[0].messageId : null;
    getPWCRDetail(mId);
  } else if (requestType === 'DCRR') {
    getDCRRDetail(referenceData);
  } else if (requestType === 'SDPC') {
    const departmentName = referenceData && referenceData.length ? referenceData[0].name : null;
    const departmentCode = referenceData && referenceData.length ? referenceData[0].code : null;
    getSDPXDetail(departmentCode, departmentName);
  } else if (requestType === 'SDPU') {
    const departmentName = referenceData && referenceData.length ? referenceData[1].name : null;
    const departmentCode = referenceData && referenceData.length ? referenceData[0] : null;
    getSDPXDetail(departmentCode, departmentName);
  } else if (requestType === 'SDPD') {
    const departmentCode = referenceData && referenceData.length ? referenceData[0] : null;
    getSDPDDetail(departmentCode);
  } else if (requestType === 'SUGC') {
    const roles = referenceData && referenceData.length ? referenceData[0].roles : [];
    const permissionIds = roles.map(r => r.id);
    const groupName = referenceData[0].name;
    const groupId = referenceData[0].id;
    getSUGXDetail(permissionIds, groupName, groupId);

  } else if (requestType === 'SUGU') {
    // Get roletree
    const roles = referenceData && referenceData.length ? referenceData[1].roles : [];
    const permissionIds = roles.map(r => r.id);
    const groupName = referenceData[1].name;
    const groupId = referenceData && referenceData.length ? referenceData[0] : null;
    getSUGXDetail(permissionIds, groupName, groupId);
  } else if (requestType === 'SUGD') {
    const id = referenceData && referenceData.length ? referenceData[0] : null;
    getSUGDDetail(id);
  } else if (requestType === 'SUSC') {
    const rData = referenceData && referenceData.length ? referenceData[0] : null;
    if (rData !== null && rData !== undefined) {
      const departmentCode = rData.departmentCode;
      const changePassword = rData.changePassword;
      const email = rData.email;
      const groups = rData.groups ? rData.groups : [];
      const groupIds = groups.map(g => g.id);
      const username = rData.username;
      const status = rData.status
      getSUSXDetail({departmentCode, changePassword, email, groupIds, username, status});
    }
  } else if (requestType === 'SUSU') {
    const rData = referenceData && referenceData.length ? referenceData[1] : null;
    if (rData !== null && rData !== undefined) {
      const departmentCode = rData.departmentCode;
      const changePassword = rData.changePassword;
      const email = rData.email;
      const groups = rData.groups ? rData.groups : [];
      const groupIds = groups.map(g => g.id);
      const username = rData.username;
      const status = rData.status;
      const userId = referenceData && referenceData.length ? referenceData[0] : null;
      getSUSXDetail({departmentCode, changePassword, email, groupIds, username, status, id: userId});
    }
  } else if (requestType === 'SUSD') {
    const userId = referenceData && referenceData.length ? referenceData[0] : null;
    getSUSDDetail(userId);
  } else if (requestType === 'SUSS') {
    const userId = referenceData && referenceData.length ? referenceData[0] : null;
    const status = referenceData && referenceData.length ? referenceData[1] : null;
    getSUSSDetail(userId, status);
  } else {
    console.error('requestType not regconized');
  }
};

const getDXULDetail = (id) => {
  GetDataSubmissionDetail(store, actions, 'uploadId', id);
};
const getCRULDetail = (id) => {
  GetCreditReportDetail(store, actions, 'uploadId', id);
};
const getPDULDetail = (id) => {
  GetPMDSDetail(store, actions, 'submission', 'uploadId', id);
};
const getPCULDetail = (id) => {
  GetPMDSDetail(store, actions, 'enquiry', 'uploadId', id);
};
const getPPCRDetail = (id) => {
  GetBusinessDetail(store, actions, 'pmds-prescribed-consent-list', id);
};
const getPWCRDetail = (id) => {
  GetBusinessDetail(store, actions, 'pmds-withdrawal-prescribed-consent-list', id);
};
const getDCRRDetail = (referenceData) => {
  const rData = referenceData ? referenceData[0] : null;
  store.dispatch(actions.setDetailLoading(true));
  const callback = () => {
    store.dispatch(actions.setDetail(rData));
    store.dispatch(actions.setDetailLoading(false));
  };
  setTimeout(callback, 500);
};

const getSUGXDetail = (permissionIds, groupName, groupId) => {
  store.dispatch(actions.setDetailLoading(true));
  const callback = () => {
    const {roleTree} = store.getState();
    const roleStates = initRoleStates(roleTree, permissionIds);
    const detail = {roleStates, groupName, groupId};
    store.dispatch(actions.setDetail(detail));
    store.dispatch(actions.setDetailLoading(false));
  };
  setTimeout(callback, 500);
};

const getSUGDDetail = (id) => {
  store.dispatch(actions.setDetailLoading(true));
  fetch(`${APIPrefix}/group/${id}`).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      const {roleTree} = store.getState();
      const roleStates = initRoleStates(roleTree, result.data.permissionIds);
      const detail = {roleStates, groupName: result.data.groupName, groupId: result.data.id};
      store.dispatch(actions.setDetail(detail));
    } else {
      displayActionResponse(result);
    }
    store.dispatch(actions.setDetailLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const getSDPXDetail = (departmentCode, departmentName) => {
  store.dispatch(actions.setDetailLoading(true));
  const callback = () => {
    const detail = {departmentCode, departmentName};
    store.dispatch(actions.setDetail(detail));
    store.dispatch(actions.setDetailLoading(false));
  };
  setTimeout(callback, 500);
};

const getSDPDDetail = (departmentCode) => {
  store.dispatch(actions.setDetailLoading(true));
  fetch(`${APIPrefix}/department/${departmentCode}`).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      store.dispatch(actions.setDetail(result.data));
    } else {
      displayActionResponse(result);
    }
    store.dispatch(actions.setDetailLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const getSUSXDetail = (params) => {
  const fetchList = [
    fetch(`${APIPrefix}/group`),
    fetch(`${APIPrefix}/department`),
  ];
  const handler = (data) => {
    let group = data[0];
    let dept = data[1];
    const detail = {
      departments: dept,
      userGroups: group,
      changePassword: params.changePassword,
      departmentCode: params.departmentCode,
      email: params.email,
      groupIds: params.groupIds,
      id: params.id,
      status: params.status,
      username: params.username,
    };
    store.dispatch(actions.setDetail(detail));
  };
  GetSUSXDetail(fetchList, handler);
};

const getSUSDDetail = (id) => {
  const fetchList = [
    fetch(`${APIPrefix}/group`),
    fetch(`${APIPrefix}/department`),
    fetch(`${APIPrefix}/user/${id}`),
  ];
  const handler = (data) => {
    let group = data[0];
    let dept = data[1];
    let user = data[2];
    const detail = {
      departments: dept,
      userGroups: group,
      changePassword: false,
      departmentCode: user.departmentCode,
      email: user.email,
      groupIds: user.groupIds,
      id: id,
      status: user.status,
      username: user.username,
    };
    store.dispatch(actions.setDetail(detail));
  };
  GetSUSXDetail(fetchList, handler);
};
const getSUSSDetail = (id, status) => {
  const fetchList = [
    fetch(`${APIPrefix}/user/${id}`),
  ];
  const handler = (data) => {
    let user = data[0];
    const detail = {
      departments: null,
      userGroups: null,
      changePassword: false,
      departmentCode: null,
      email: null,
      groupIds: null,
      id,
      status,
      originalStatus: user.status,
      username: user.username,
    };
    store.dispatch(actions.setDetail(detail));
  };
  GetSUSXDetail(fetchList, handler);
};

const GetSUSXDetail = (fetchList, respHandler) => {
  store.dispatch(actions.setDetailLoading(true));
  Promise.all(fetchList).then(responses => {
    return Promise.all(responses.map(resp => {
      if (resp.redirected) {
        window.location.href = resp.url;
      } else if (resp.status === 403) {
        alert('Your session has expired. Please login again.');
        window.location.href = '/login';
      } else {
        return resp.json();
      }
    }));
  }).then(results => {
    let data = [];
    results.forEach(result => {
      if (result.result === "Success") {
        data.push(result.data);
      } else {
        displayActionResponse(result);
        throw new Error("Failed to get data from response json");
      }
    });
    respHandler(data);
    store.dispatch(actions.setDetailLoading(false));
  }).catch(error => {
    console.error(error);
  });

}

const getFlattenRoleTree = () => {
  GetFlattenRoleTree(store, actions);
};

const downloadApprovalFile = (requestType, id, filename) => {
  const idType = 'uploadId';
  let url;
  if (requestType === 'DSUL' || requestType === 'DAUL' || requestType === 'IDUL') {
    url = `${APIPrefix}/crp-service/data-submission/detail/${idType}/${id}/file`;
  } else if (requestType === 'CRUL') {
    url = `${APIPrefix}/crp-service/credit-report/detail/${idType}/${id}/file`;
  } else if (requestType === 'PDUL') {
    url = `${APIPrefix}/crp-service/pmds/submission/detail/${idType}/${id}/file`;
  } else if (requestType === 'PCUL') {
    url = `${APIPrefix}/crp-service/pmds/enquiry/detail/${idType}/${id}/file`;
  }
  fetch(url).then(response => {
    handleDownloadResponse(response, filename);
  }).catch(error => {
    console.error(error);
  });
};
