"use strict";

import {createStore, displayActionResponse} from '../common.js';
import {APIPrefix, getJsonRequestHeaderWithCsrf} from '../common.js';

// Redux architecture pieces
export const actions = {
  getItems: () => {
    // Get items from fetch
    return {
      type: 'GET_ITEMS'
    };
  },
  setItems: (items) => {
    // To be called by getItems callback
    return {
      type: 'SET_ITEMS',
      payload: items,
    };
  },
  setChangeItem: (item) => {
    return {
      type: 'SET_CHANGE_ITEM',
      payload: item,
    };
  },
  resetChangeItem: () => {
    return {
      type: 'RESET_CHANGE_ITEM',
    };
  },
  updateItem: (id, status) => {
    return {
      type: 'UPDATE_ITEM',
      payload: {
        id,
        status,
      }
    };
  },
  deleteItem: (id) => {
    return {
      type: 'DELETE_ITEM',
      payload: id,
    };
  },
  createItem: () => {
    return {
      type: 'CREATE_ITEM',
    };
  },
};

const initialState = {
  items: [],
  changeItem: {},
};

const accmgmtSigningReducer = (state = initialState, action) => {
  let changeItem
  switch (action.type) {
    case 'GET_ITEMS':
      getSigningItems();
      return state;

    case 'SET_ITEMS':
      // Set items & initialize page + currentPageItems
      return Object.assign(Object.assign({}, state), {
        items: action.payload,
      });

    case 'SET_CHANGE_ITEM':
      changeItem = Object.assign(
         Object.assign({}, state.changeItem), action.payload);
      return Object.assign(Object.assign({}, state), {changeItem});

    case 'RESET_CHANGE_ITEM':
      changeItem = Object.assign({}, initialState.changeItem);
      return Object.assign(Object.assign({}, state), {changeItem});

    case 'UPDATE_ITEM':
      updateSigning(action.payload.id, action.payload.status);
      return state;

    case 'DELETE_ITEM':
      deleteSigning(action.payload);
      return state;

    case 'CREATE_ITEM':
      postSigning(state.changeItem);
      return state;

    default:
      return state;
  }
};

export const store = createStore(accmgmtSigningReducer);

const getSigningItems = (params) => {
  const url = `${APIPrefix}/secret/signing`;
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      store.dispatch(actions.setItems(result.data));
    } else {
      displayActionResponse(result);
    }
  }).catch(error => {
    console.error(error);
  });
};

const postSigning = (changeItem) => {
  const body = Object.assign({}, changeItem);
  fetch(`${APIPrefix}/secret/signing`, {
    method: 'POST',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const deleteSigning = (id) => {
  fetch(`${APIPrefix}/secret/signing/${id}`, {
    method: 'DELETE',
    headers: getJsonRequestHeaderWithCsrf(),
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const updateSigning = (id, status) => {
  const body = Object.assign({}, {status});
  fetch(`${APIPrefix}/secret/signing/${id}`, {
    method: 'PUT',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else if (response.status === 403) {
      alert('Your session has expired. Please login again.');
      window.location.href = '/login';
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};
