"use strict";

import {
  checkInvalidCharacterInItem,
  checkRequiredFieldInItem,
  checkStatusColor,
  convertDatetimeToLocaleString,
  displayDropdownBoxLabel,
  getFilterDateInputValue,
  handleCollapseCardHeaderOnClick,
  handleDropDownOnClick,
  handleInputLabels,
  handleModalOnToggleById,
  hasInvalidCharacter,
  toggleLoadingIcon
} from './common.js';

import {getBaseUploadForm} from './component/upload_form.js';
import {getHistoryTable} from './component/history_table.js';
import {
  getHistoryModal,
  renderCRAResponse,
  renderCRPResponse
} from './component/history_modal.js';

export const History = (store, actions, tabname) => {
  const inst = {};
  inst.tag = store.getState().tag;
  inst.store = store;
  inst.actions = actions;
  inst.store.subscribe((action) => {
    const {tab} = inst.store.getState();
    if (tabname && tab !== tabname) {
      return;
    }
    if (action.type === 'SET_RECORDS') {
      const {records} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='table']`);
      inst.table.buildTable(container, records);
    } else if (action.type === 'SET_TOTAL_PAGES') {
      const {totalPages, page, size} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='table']`);
      inst.table.buildPagination(container, page, size, totalPages);
    } else if (action.type === 'SET_TOTAL_RECORDS') {
      const {totalRecords} = inst.store.getState();
      inst.table.setTotalRecordCount(totalRecords);
    } else if (action.type === 'SET_DETAIL') {
      const {detail} = inst.store.getState();
      inst.modal.buildModalContent(detail);
      inst.modal.buildModalFooter(detail);
      const isIdTypeMessageId = !!detail.messageId;
      if (isIdTypeMessageId) {
        inst.buildResolveModal(detail.messageId, detail.resolved);
      }
    } else if (action.type === 'SET_DETAIL_LOADING') {
      const {detailLoading} = inst.store.getState();
      const modalContainer = inst.modal.getContainer();
      toggleLoadingIcon(modalContainer, detailLoading)
    }
  });
  inst.buildFilter = () => {
    const queryParams = ['messageId', 'fileName', 'targetedCRA', 'channel', 'status', 'username', 'originalFilename',
                          'createDatetimeFrom', 'createDatetimeTo', 'messageRemark', 'excludeResolved', 'departmentCode'];
    const handleQueryInputOnChange = (name, target) => {
      let value;
      if (target.tagName === 'INPUT') {
        if (target.type === 'text') {
          value = target.value;
        } else if (target.type === 'checkbox') {
          value = target.checked;
        } else if (target.type === 'datetime-local' || target.type === 'date') {
          value = getFilterDateInputValue(name, target.value);
        }
      } else if (target.tagName === 'SELECT') {
        value = target.value;
      }
      const card = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='filter']`);
      const labelContainer = card.querySelector(`[data-${inst.tag}-filter='${name}']`);
      handleInputLabels(labelContainer, value, target.tagName);
      inst.store.dispatch(inst.actions.setQuery({[name]: value}));
    };
    const handleSearchButtonOnClick = () => {
      const {recordLoading} = inst.store.getState();
      if (recordLoading) {
        return;
      }
      if (!inst.isSearchButtonActive()) {
        return;
      }
      inst.store.dispatch(inst.actions.setPage());
    };
    const handleClearButtonOnClick = () => {
      const {recordLoading} = inst.store.getState();
      if (recordLoading) {
        return;
      }
      queryParams.forEach(q => {
        const fieldContainer = card.querySelector(`[data-${inst.tag}-filter='${q}']`);
        const input = fieldContainer.querySelector("input");
        if (input) {
          if (input.type === 'checkbox') {
            input.checked = false;
          } else {
            input.value = null;
          }
        }
        const select = fieldContainer.querySelector("select");
        if (select) {
          select.selectedIndex = -1;
        }
      });
      inst.store.dispatch(inst.actions.clearQuery());
      inst.store.dispatch(inst.actions.getRecords());
    };
    const buildCollapseCard = (card) => {
      const header = card.querySelector("[data-collapsecard='header']");
      header.addEventListener('click', () => handleCollapseCardHeaderOnClick(card));
    };
    const renderFilterContent = (card) => {
      queryParams.forEach(q => {
        const fieldContainer = card.querySelector(`[data-${inst.tag}-filter='${q}']`);
        const input = fieldContainer.querySelector("input");
        if (input) {
          input.addEventListener("change", (e) => handleQueryInputOnChange(q, e.target));
        }
        const select = fieldContainer.querySelector("select");
        if (select) {
          select.selectedIndex = -1;
          select.addEventListener("change", (e) => handleQueryInputOnChange(q, e.target));
        }
      });
    };
    inst.isSearchButtonActive = () => {
      const {query} = store.getState();
      // let isActive = false;
      // isActive = checkHasOneFieldInItem(query);
      const checkInvalidCharacterFields = ["messageId", "fileName", "username", "messageRemark", "departmentCode", "originalFilename"];
      let isActive = checkInvalidCharacterInItem(query, checkInvalidCharacterFields);
      return isActive;
    };
    const card = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='filter']`);
    buildCollapseCard(card);
    renderFilterContent(card);
    const searchButton = card.querySelector("#history-filter-search");
    searchButton.addEventListener("click", handleSearchButtonOnClick);
    // if (searchButton) {
    //   searchButton.disabled = !inst.isSearchButtonActive();
    // }
    const clearButton = card.querySelector("#history-filter-clear");
    clearButton.addEventListener("click", handleClearButtonOnClick);
  };
  inst.table = Object.create(getHistoryTable());
  inst.setTable = () => {
    inst.table.fillHistoryTableRow = function(cols, rec) {
      // Handle messageId
      const messageIdElement = document.createElement('a');
      const hasMessageId = !!rec.messageId;
      messageIdElement.textContent = hasMessageId ? rec.messageId : `U${rec.id}`;
      messageIdElement.classList.add("font-medium");
      messageIdElement.classList.add("text-blue-600");
      messageIdElement.classList.add("hover:underline");
      const modalOnToggleParams = {
        id: hasMessageId ? rec.messageId : `${rec.id}`,
        idType: hasMessageId ? 'messageId' : 'uploadId',
      };
      messageIdElement.addEventListener('click', () => inst.modal.handleHistoryModalOnToggle(modalOnToggleParams));
      cols[0].appendChild(messageIdElement);
      cols[1].textContent = rec.actionType;
      cols[2].textContent = rec.originalFilename;
      cols[3].textContent = rec.recordCount;
      cols[4].textContent = rec.channel;
      cols[5].textContent = rec.user;
      cols[6].textContent = rec.departmentCode;
      cols[7].textContent = rec.targetedCRAs.join(', ');
      cols[8].textContent = convertDatetimeToLocaleString(rec.createDatetime);
      // Handle Status
      const statusSpan = document.createElement('span');
      const statusTextColor = checkStatusColor(rec.status);
      statusSpan.classList.add(`text-${statusTextColor}`);
      statusSpan.innerHTML = rec.status;
      cols[9].appendChild(statusSpan);
      let resolvedIcon;
      if (rec.resolved) {
        resolvedIcon = document.getElementById("history-table-resolved-icon").cloneNode(true).content.firstElementChild;
        cols[9].appendChild(resolvedIcon);
      }
      cols[10].textContent = rec.messageRemark;
    };
    inst.table.setPage = function(page) {
      inst.store.dispatch(inst.actions.setPage(page))
    };
    inst.table.setSize = function(size) {
      inst.store.dispatch(inst.actions.setSize(size))
    };
  };
  inst.modal = Object.create(getHistoryModal());
  inst.setModal = () => {
    const {tab} = inst.store.getState();
    const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector("#detail-modal");
    inst.modal.setContainer(container);
    const template = document.getElementById(`${tab}-modal-content-template`);
    inst.modal.setTemplate(template);
    inst.modal.onActivate = function(params) {
      inst.store.dispatch(inst.actions.getDetail(params.idType, params.id));
    };
    inst.modal.renderModalContent = function(content, detail) {
      const fields = ["messageId", "actionType", "channel", "user", "departmentCode", "approvedUserName",
        "filename", "originalFilename", "fileVersion", "fileSize", "fileAction",
        "recordCount", "targetedCRAs", "status", "messageRemark",
        "errorMessage", "resolvedMessage"];
      fields.forEach(f => {
        const input = content.querySelector(`[data-${inst.tag}-detail='${f}']`).querySelector("input");
        if (input && detail[f]) {
          input.value = detail[f];
        }
      });

      const datetimeInput = content.querySelector(`[data-${inst.tag}-detail='createDatetime']`).querySelector("input");
      datetimeInput.value = convertDatetimeToLocaleString(detail.createDatetime);

      const hasMessageId = !!detail.messageId;

      // Handle download button
      const filenameContainer = content.querySelector(`[data-${inst.tag}-detail='filename']`);
      const originalFilenameContainer = content.querySelector(`[data-${inst.tag}-detail='originalFilename']`);
      let downloadButtonContainer;
      if (hasMessageId) {
        originalFilenameContainer.querySelector(`[data-${inst.tag}-detail='download']`)?.remove();
        downloadButtonContainer = filenameContainer;
      } else {
        filenameContainer.querySelector(`[data-${inst.tag}-detail='download']`)?.remove();
        downloadButtonContainer = originalFilenameContainer;
      }

      let params = {
        idType: hasMessageId ? 'messageId' : 'uploadId',
        id: hasMessageId ? detail.messageId : detail.id,
        filename: detail.filename ? detail.filename : detail.originalFilename,
      };
      const downloadButton = downloadButtonContainer.querySelector(`[data-${inst.tag}-detail='download']`);
      downloadButton.addEventListener('click', () => inst.store.dispatch(inst.actions.getFile(params.idType, params.id, params.filename)));

      const crpResp = content.querySelector(`[data-${inst.tag}-detail='crpResponse']`);
      renderCRPResponse(inst.tag, crpResp, detail.crpResponse);

      // craResponses
      const craRespList = content.querySelector(`[data-${inst.tag}-detail='craResponseList']`);
      craRespList.innerHTML = "";

      if (detail.craResponses && detail.craResponses.length) {
        detail.craResponses.forEach(response => {
          renderCRAResponse(inst.tag, craRespList, response);
        });
      } else {
        const noRecord = document.getElementById('history-modal-no-resp').cloneNode(true).content;
        craRespList.appendChild(noRecord);
      }
      // Credit reports
      const renderCreditReport = (list, resp) => {
        const credrptTemplate = document.getElementById(`history-modal-${inst.tag}-credrpt`);
        const content = credrptTemplate.cloneNode(true).content;
        const sourceInput = content.querySelector(`[data-${inst.tag}-detail='generatedCRA'`).querySelector("input");
        sourceInput.value = resp.generatedCRA;
        const createDatetimeInput = content.querySelector(`[data-${inst.tag}-detail='createDatetime'`).querySelector("input");
        createDatetimeInput.value = convertDatetimeToLocaleString(resp.createDatetime);
        const messageIdInput = content.querySelector(`[data-${inst.tag}-detail='messageId'`).querySelector("input");
        messageIdInput.value = resp.messageId;
        const reportArchiveInput = content.querySelector(`[data-${inst.tag}-detail='reportFileName'`).querySelector("input");
        reportArchiveInput.value = resp.reportFileName;
        const archiveDownload = content.querySelector(`[data-${inst.tag}-detail='archiveDownload']`);
        archiveDownload.addEventListener('click', () => inst.store.dispatch(inst.actions.getArchive(resp.messageId, resp.reportFileName)));
        const reportContainer = content.querySelector(`[data-${inst.tag}-detail='reportContainer']`);
        const report = reportContainer.querySelector(`[data-${inst.tag}-detail='innerFileName']`);
        const reportList = report.parentNode;
        if (resp.innerFileNames && resp.innerFileNames.length) {
          resp.innerFileNames.forEach((f, idx) => {
            if (idx > 0) {
              reportList.appendChild(report.cloneNode(true));
            }
          });
          resp.innerFileNames.forEach((f, idx) => {
            const innerFileNameInput = reportList.children[idx].querySelector("input");
            innerFileNameInput.value = f;
            const reportDownload = reportList.children[idx].querySelector("button");
            reportDownload.addEventListener('click', () => inst.store.dispatch(inst.actions.getReport(resp.messageId, f)));
          });
        } else {
          reportContainer.remove();
        }
        list.appendChild(content);
      };
      const credrptList = content.querySelector(`[data-${inst.tag}-detail='creditReportList']`);
      credrptList.innerHTML = "";
      if (detail.creditReports && detail.creditReports.length) {
        detail.creditReports.forEach(response => {
          renderCreditReport(credrptList, response);
        });
      } else {
        const noRecord = document.getElementById('history-modal-no-resp').cloneNode(true).content;
        credrptList.appendChild(noRecord);
      }
    };
    inst.modal.renderModalFooter = function(container, detail) {
      const oldResolveButton = container.querySelector(`[data-${inst.tag}-detail='resolve']`).querySelector("button");
      const resolveButton = oldResolveButton.cloneNode(true);
      if (detail.resolved || !detail.messageId) {
        resolveButton.disabled = true;
      } else {
        resolveButton.disabled = false;
        const resolveModal = document.getElementById("resolve-modal");
        resolveButton.addEventListener('click', () => handleModalOnToggleById("resolve-modal"));
      }
      oldResolveButton.replaceWith(resolveButton);
    };
  };
  inst.buildResolveModal = (messageId, resolved) => {
    const resolveModal = document.getElementById("resolve-modal");
    const resolvedMessageContainer = resolveModal.querySelector(`[data-${inst.tag}-resolve='resolvedMessage']`);
    // Set event for submit button
    const handleResolveSubmitButtonOnClick = () => {
      if (!isResolveButtonActive()) {
        return;
      }
      inst.store.dispatch(inst.actions.resolveMessage(messageId));
    };
    const isResolveButtonActive = (value = "") => {
      if (resolved) {
        alert("Message already resolved");
      }
      let isActive = resolved;
      if (value) {
        if (hasInvalidCharacter(value)) {
          alert("Invalid character in resolve message");
          return false;
        }
      }
      return true
    };
    const handleResolveMessageOnChange = (value) => {
      handleInputLabels(resolvedMessageContainer, value, "TEXTAREA");
      inst.store.dispatch(inst.actions.setResolvedMessage(value));
    };
    const resolveButton = resolveModal.querySelector(`[data-${inst.tag}-resolve='submit']`).querySelector("button");
    const newResolveButton = resolveButton.cloneNode(true);
    newResolveButton.addEventListener("click", handleResolveSubmitButtonOnClick);
    resolveButton.replaceWith(newResolveButton); //Remove event handler

    const resolveTextarea = resolvedMessageContainer.querySelector("input");
    const newResolveTextarea = resolveTextarea.cloneNode(true);
    newResolveTextarea.addEventListener("change", (e) => handleResolveMessageOnChange(e.target.value));
    resolveTextarea.replaceWith(newResolveTextarea);
  };
  inst.initialize = () => {
    const {tag} = inst.store.getState();
    inst.tag = tag;
    inst.buildFilter();
    inst.setModal();
    inst.setTable();
    store.dispatch(inst.actions.getRecords());
  };
  // Table related method
  return inst;
};

export const Enquiry = (crStore, crActions) => {
  const inst = getBaseUploadForm('enquiry', crStore, crActions);
  inst.renderFormItem = function(content) {
    const targetedCRAsSelect = content.querySelector(`[data-${this.tag}-${this.tab}='targetedCRAs']`);
    targetedCRAsSelect.querySelector("[data-dropdown='box']").addEventListener('click', () => handleDropDownOnClick(targetedCRAsSelect));
    targetedCRAsSelect.querySelector("[data-dropdown='mask']").addEventListener('click', () => handleDropDownOnClick(targetedCRAsSelect));
    const targetedCRAsInputs = targetedCRAsSelect.querySelector("[data-dropdown='container']").querySelectorAll('input');
    targetedCRAsInputs.forEach(input => {
      input.addEventListener('change', () => displayDropdownBoxLabel(targetedCRAsSelect));
      input.addEventListener('change', () => this.handleUploadItemOnChangeByValue('targetedCRAs', Array.from(targetedCRAsInputs).filter(i => i.checked).map(o => o.value)));
    });

    const messageRemarkInput = content.querySelector(`[data-${this.tag}-${this.tab}='messageRemark']`).querySelector("input");
    messageRemarkInput.addEventListener('change', (e) => this.handleUploadItemOnChange('messageRemark', e.target));
    const fileFormatSelect = content.querySelector(`[data-${this.tag}-${this.tab}='fileFormat']`).querySelector("select");
    fileFormatSelect.addEventListener('change', (e) => this.handleUploadItemOnChange('fileFormat', e.target));
    fileFormatSelect.selectedIndex = -1;
    const dataFileInput = content.querySelector(`[data-${this.tag}-${this.tab}='dataFile']`).querySelector("input");
    dataFileInput.addEventListener('change', (e) => this.handleUploadItemOnChange('dataFile', e.target));
  };
  inst.isFormSubmittable = function() {
    const {uploadItem} = this.store.getState();
    let isValid = checkRequiredFieldInItem(uploadItem, ['fileFormat', 'dataFile', 'targetedCRAs']);
    isValid = isValid && checkInvalidCharacterInItem(uploadItem, ['messageRemark']);
    return isValid;
  };
  return inst;
};
