"use strict";

import {handleCollapseCardHeaderOnClick} from './common.js';
import {handleDropDownOnClick, displayDropdownBoxLabel} from './common.js';
import {hasInvalidCharacter, isNumeric, isEmptyField} from './common.js';
import {checkRequiredFieldInItem, checkInvalidCharacterInItem} from './common.js';
import {handleInputLabels, handleModalOnToggleById, toggleLoadingIcon} from './common.js';
import {convertDatetimeToLocaleString, getFilterDateInputValue, checkStatusColor} from './common.js';
import {getFileExtensionFromMimeType} from './common.js';
import {getHistoryTable} from './component/history_table.js';
import {getHistoryModal, renderCRAResponse, renderRepliedMessageResponse} from './component/history_modal.js';
import {renderReminderMessageResponse} from './component/history_modal.js';
import {getDetailTypeFromMessageType} from './model/business_model.js';

export const History = (store, actions, tabname) => {
  const inst = {};
  inst.store = store;
  inst.actions = actions;
  inst.tag = store.getState().tag;
  inst.store.subscribe((action) => {
    const {tab} = inst.store.getState();
    if (tabname && tab !== tabname) {
      return;
    }
    if (action.type === 'SET_RECORDS') {
      const {records} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='table']`);
      inst.table.buildTable(container, records);
    } else if (action.type === 'SET_TOTAL_PAGES') {
      const {totalPages, page, size} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='table']`);
      inst.table.buildPagination(container, page, size, totalPages);
    } else if (action.type === 'SET_TOTAL_RECORDS') {
      const {totalRecords} = inst.store.getState();
      inst.table.setTotalRecordCount(totalRecords);
    } else if (action.type === 'SET_DETAIL') {
      const {detail, detailType} = inst.store.getState();
      if (detailType === 'data-file') {
        inst.dataFileModal.buildModalContent(detail);
        inst.dataFileModal.buildModalFooter(detail);
      } else if (detailType === 'dcr') {
        inst.dcrModal.buildModalContent(detail);
        inst.dcrModal.buildModalFooter(detail);
        inst.buildReplyModal(detail.messageId);
      } else if (detailType === 'pmds-prescribed-consent-list') {
        inst.pmdscstModal.buildModalContent(detail);
        inst.pmdscstModal.buildModalFooter(detail);
        inst.buildReplyModal(detail.messageId);
      } else if (detailType === 'pmds-withdrawal-prescribed-consent-list') {
        inst.pmdswdlModal.buildModalContent(detail);
        inst.pmdswdlModal.buildModalFooter(detail);
        inst.buildReplyModal(detail.messageId);
      }
    } else if (action.type === 'SET_RECORD_LOADING') {
      const {tab, recordLoading} = inst.store.getState();
      const table = document.querySelector(`[data-${inst.tag}-${tab}='table']`);
      const tableLoading = table?.querySelector("[data-table='loading']");
      toggleLoadingIcon(tableLoading, recordLoading);
    } else if (action.type === 'SET_REPLY_LOADING') {
      const {tab, replyLoading} = inst.store.getState();
      const handleReplyLoading = () => {
        const replyModal = document.getElementById("reply-modal");
        const replyModalLoading = replyModal?.querySelector(`[data-${inst.tag}-reply='loading']`);
        const replyAcceptButton = replyModal?.querySelector(`[data-${inst.tag}-reply='accept']`)?.querySelector("button");
        const replyRejectButton = replyModal?.querySelector(`[data-${inst.tag}-reply='reject']`)?.querySelector("button");
        if (replyAcceptButton) {
          replyAcceptButton.disabled = true;
        }
        if (replyRejectButton) {
          replyRejectButton.disabled = true;
        }
        toggleLoadingIcon(replyModalLoading, replyLoading);
      };
      handleReplyLoading();
    } else if (action.type === 'SET_DETAIL_LOADING') {
      const {detailLoading, detailType} = inst.store.getState();
      const handleDetailLoading = () => {
        let modalContainer;
        if (detailType === 'data-file') {
          modalContainer = inst.dataFileModal.getContainer();
        } else if (detailType === 'dcr') {
          modalContainer = inst.dcrModal.getContainer();
        } else if (detailType === 'pmds-prescribed-consent-list') {
          modalContainer = inst.pmdscstModal.getContainer();
        } else if (detailType === 'pmds-withdrawal-prescribed-consent-list') {
          modalContainer = inst.pmdswdlModal.getContainer();
        }
        if (modalContainer) {
          toggleLoadingIcon(modalContainer, detailLoading);
        }
      };
      handleDetailLoading();
    }
  });
  inst.table = Object.create(getHistoryTable());
  inst.setTable = () => {
    inst.table.fillHistoryTableRow = function(cols, rec) {
      // Handle messageId
      const messageIdElement = document.createElement('a');
      const detailType = getDetailTypeFromMessageType(rec.messageType);
      messageIdElement.textContent = rec.messageId;
      if (detailType) {
        messageIdElement.classList.add("font-medium");
        messageIdElement.classList.add("text-blue-600");
        messageIdElement.classList.add("hover:underline");
        const modalOnToggleParams = {
          messageType: rec.messageType,
          messageId: rec.messageId,
        };
        if (detailType === 'data-file') {
          messageIdElement.addEventListener('click', () => inst.dataFileModal.handleHistoryModalOnToggle(modalOnToggleParams));
        } else if (detailType === 'dcr') {
          messageIdElement.addEventListener('click', () => inst.dcrModal.handleHistoryModalOnToggle(modalOnToggleParams));
        } else if (detailType === 'pmds-prescribed-consent-list') {
          messageIdElement.addEventListener('click', () => inst.pmdscstModal.handleHistoryModalOnToggle(modalOnToggleParams));
        } else if (detailType === 'pmds-withdrawal-prescribed-consent-list') {
          messageIdElement.addEventListener('click', () => inst.pmdswdlModal.handleHistoryModalOnToggle(modalOnToggleParams));
        }
      }
      cols[0].appendChild(messageIdElement);

      const msgtypeSpan = document.getElementById(`${inst.tag}-history-span-template`).cloneNode(true).content;
      Array.from(msgtypeSpan.children).filter(c => c.getAttribute(`data-${inst.tag}-span`) !== detailType).forEach(elem => {
        elem.remove();
      });
      if (msgtypeSpan.children.length) {
        cols[1].appendChild(msgtypeSpan);
      } else {
        cols[1].textContent = rec.messageType;
      }
      cols[2].textContent = rec.source;
      cols[3].textContent = convertDatetimeToLocaleString(rec.createDatetime);
      cols[4].textContent = convertDatetimeToLocaleString(rec.lastUpdateDatetime);
      // Handle Status
      const statusSpan = document.createElement('span');
      const statusTextColor = checkStatusColor(rec.status);
      if (statusTextColor) {
        statusSpan.classList.add(`text-${statusTextColor}`);
      }
      statusSpan.innerHTML = rec.status;
      cols[5].appendChild(statusSpan);
      let resolvedIcon;
      if (rec.resolved) {
        resolvedIcon = document.getElementById("history-table-resolved-icon").cloneNode(true).content.firstElementChild;
        cols[5].appendChild(resolvedIcon);
      }
      cols[6].textContent = rec.messageRemark;
    };
    inst.table.setPage = function(page) {
      inst.store.dispatch(inst.actions.setPage(page))
    };
    inst.table.setSize = function(size) {
      inst.store.dispatch(inst.actions.setSize(size))
    };
  };
  inst.dataFileModal = Object.create(getHistoryModal());
  inst.getDataFileModalSetter = (modal, templateName, fields) => {
    return () => {
      const {tab} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector("#detail-modal");
      modal.setContainer(container);
      const template = document.getElementById(`datafile-modal-content-template`);
      modal.setTemplate(template);
      modal.onActivate = function(params) {
        inst.store.dispatch(inst.actions.getDetail(params.messageType, params.messageId));
      };
      modal.renderModalContent = function(content, detail) {
        fields.forEach(f => {
          const input = content.querySelector(`[data-${inst.tag}-detail='${f}']`).querySelector("input");
          if (detail[f]) {
            input.value = detail[f];
          }
        });

        const msgtypeSpan = document.getElementById(`${inst.tag}-history-span-template`).cloneNode(true).content;
        const detailType = getDetailTypeFromMessageType(detail.messageType);
        const messageTypeInput = content.querySelector(`[data-${inst.tag}-detail='messageType']`).querySelector("input");
        messageTypeInput.value = Array.from(msgtypeSpan.children).find(c => c.getAttribute(`data-${inst.tag}-span`) === detailType).textContent;

        const datetimeInput = content.querySelector(`[data-${inst.tag}-detail='createDatetime']`).querySelector("input");
        datetimeInput.value = convertDatetimeToLocaleString(detail.createDatetime);

        let fileName = detail.filename ? detail.filename : `${detail.messageId}_${detail.source}_response`;
        const downloadButton = content.querySelector(`[data-${inst.tag}-detail='download']`);
        downloadButton.addEventListener('click', () => inst.store.dispatch(inst.actions.getFile(detail.messageId, fileName)));

        // Handle CRA Response
        const craRespList = content.querySelector(`[data-${inst.tag}-detail='craResponseList']`);
        craRespList.innerHTML = "";

        if (detail.craResponses && detail.craResponses.length) {
          detail.craResponses.forEach(response => {
            renderCRAResponse(inst.tag, craRespList, response);
          });
        } else {
          const noRecord = document.getElementById('history-modal-no-resp').cloneNode(true).content;
          craRespList.appendChild(noRecord);
        }
      };
      modal.renderModalFooter = function(container, detail) {}
    }
  }
  inst.setDataFileModal = inst.getDataFileModalSetter(
    inst.dataFileModal,
    'datafile-modal-content-template',
    ["messageId", "source", "filename", "fileVersion", "fileSize", "fileAction", "dcrReferenceId", "dcrInitiator", "status"],
  )
  inst.dcrModal = Object.create(getHistoryModal());
  inst.getDCRModalSetter = (modal, modalDetailType, fields) => {
    const templateName = `${modalDetailType}-modal-content-template`;
    let replyModalTemplateId;
    if (modalDetailType === 'dcr') {
      replyModalTemplateId = 'dcr-reply-modal-template' ;
    } else {
      replyModalTemplateId = 'pmds-reply-modal-template' ;
    }
    return () => {
      const {tab} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector("#detail-modal");
      modal.setContainer(container);
      const template = document.getElementById(templateName);
      modal.setTemplate(template);
      modal.onActivate = function(params) {
        inst.store.dispatch(inst.actions.getDetail(params.messageType, params.messageId));
      };
      modal.renderModalContent = function(content, detail) {
        fields.forEach(f => {
          const input = content.querySelector(`[data-${inst.tag}-detail='${f}']`)?.querySelector("input");
          if (detail[f] && input) {
            input.value = detail[f];
          }
        });
        const msgtypeSpan = document.getElementById(`${inst.tag}-history-span-template`).cloneNode(true).content;
        const detailType = getDetailTypeFromMessageType(detail.messageType);
        const messageTypeInput = content.querySelector(`[data-${inst.tag}-detail='messageType']`).querySelector("input");
        messageTypeInput.value = Array.from(msgtypeSpan.children).find(c => c.getAttribute(`data-${inst.tag}-span`) === detailType).textContent;

        const dcrCreationDatetimeInput = content.querySelector(`[data-${inst.tag}-detail='dcrCreationDatetime']`)?.querySelector("input");
        if (dcrCreationDatetimeInput) {
          dcrCreationDatetimeInput.value = convertDatetimeToLocaleString(detail.dcrCreationDatetime);
        }

        const lastUpdateDatetimeInput = content.querySelector(`[data-${inst.tag}-detail='lastUpdateDatetime']`)?.querySelector("input");
        if (lastUpdateDatetimeInput) {
          lastUpdateDatetimeInput.value = convertDatetimeToLocaleString(detail.lastUpdateDatetime);
        }

        const datetimeInput = content.querySelector(`[data-${inst.tag}-detail='createDatetime']`)?.querySelector("input");
        if (datetimeInput) {
          datetimeInput.value = convertDatetimeToLocaleString(detail.createDatetime);
        }

        let fileName = detail.fileName ? detail.fileName : `${detail.messageId}_${detail.source}_response`;
        const fileExtension = getFileExtensionFromMimeType(detail.dataFormat);
        if (fileExtension) {
          fileName = fileName + `.${fileExtension}`;
        }
        const fileNameInput = content.querySelector(`[data-${inst.tag}-detail='fileName']`)?.querySelector("input");
        if (fileNameInput) {
          fileNameInput.value = fileName;
        }
        const downloadButton = content.querySelector(`[data-${inst.tag}-detail='download']`);
        if (downloadButton) {
          downloadButton.addEventListener('click', () => inst.store.dispatch(inst.actions.getFile(detail.messageId, fileName)));
        }

        const repliedMsgList = content.querySelector(`[data-${inst.tag}-detail='repliedMessageList']`);
        if (repliedMsgList) {
          repliedMsgList.innerHTML = "";
          const repliedMsgType = detailType === 'dcr' ? 'repliedmsg' : 'pmdsrepliedmsg';
          if (detail.repliedMessages && detail.repliedMessages.length) {
            detail.repliedMessages.forEach(response => {
              renderRepliedMessageResponse(inst.tag, repliedMsgList, response, repliedMsgType);
            });
          } else {
            const noRecord = document.getElementById('history-modal-no-resp').cloneNode(true).content;
            repliedMsgList.appendChild(noRecord);
          }
        }

        const dcrCompletionList = content.querySelector(`[data-${inst.tag}-detail='dcrCompletionNotification']`);
        if (dcrCompletionList) {
          dcrCompletionList.innerHTML = "";
          if (detail.dcrCompletionNotification) {
            renderCRAResponse(inst.tag, dcrCompletionList, detail.dcrCompletionNotification);
          } else {
            const noRecord = document.getElementById('history-modal-no-resp').cloneNode(true).content;
            dcrCompletionList.appendChild(noRecord);
          }
        }
        const reminderMessageList = content.querySelector(`[data-${inst.tag}-detail='reminderMessageList']`);
        if (reminderMessageList) {
          reminderMessageList.innerHTML = "";
          if (detail.reminderMessages && detail.reminderMessages.length) {
            detail.reminderMessages.forEach(response => {
              renderReminderMessageResponse(inst.tag, reminderMessageList, response);
            });
          } else {
            reminderMessageList.parentElement.remove();
          }
        }

        // Build Reply Modal from template
        const replyModalContainer = document.querySelector(`[data-${inst.tag}-history='reply']`);
        const replyModalTemplate = document.getElementById(replyModalTemplateId);
        replyModalContainer.innerHTML = "";
        if (replyModalTemplate) {
          replyModalContainer.appendChild(replyModalTemplate.cloneNode(true).content);
        }
      };
      modal.renderModalFooter = function(container, detail) {
        const replyContainer = container.querySelector(`[data-${inst.tag}-detail='reply']`);
        if (replyContainer.classList.contains("hidden")) {
          replyContainer.classList.remove("hidden");
        }
        const oldReplyButton = replyContainer.querySelector("button");
        const replyButton = oldReplyButton.cloneNode(true);
        if (detail.status === 'ACTION_PENDING' && detail.messageId) {
          replyButton.disabled = false;
          const replyModal = document.getElementById("reply-modal");
          replyButton.addEventListener('click', () => handleModalOnToggleById("reply-modal"));
        } else {
          replyButton.disabled = true;
          replyContainer.classList.add("hidden");
        }
        oldReplyButton.replaceWith(replyButton);
      };
    };
  };
  inst.setDCRModal = inst.getDCRModalSetter(
    inst.dcrModal,
    'dcr',
    ["messageId", "source", "dcrReferenceId", "dcrInitiator",
     "fileVersion", "fileSize", "fileAction", "dataFormat", "status"]);
  inst.pmdscstModal = Object.create(getHistoryModal());
  inst.setPMDSCSTModal = inst.getDCRModalSetter(
    inst.pmdscstModal,
    'pmds-prescribed-consent-list',
    ["messageId", "source", "fileFormat", "fileSize", "status"]);
  inst.pmdswdlModal = Object.create(getHistoryModal());
  inst.setPMDSWDLModal = inst.getDCRModalSetter(
    inst.pmdswdlModal,
    'pmds-withdrawal-prescribed-consent-list',
    ["messageId", "source", "fileFormat", "fileSize", "status"]);
  inst.buildReplyModal = (messageId) => {
    const replyModal = document.getElementById("reply-modal");
    const replyDataContainer = replyModal.querySelector(`[data-${inst.tag}-reply='replyData']`);
    replyDataContainer?.classList.remove('hidden');
    const handleReplyDataOnChange = (value) => {
      handleInputLabels(replyDataContainer, value, "TEXTAREA");
      inst.store.dispatch(inst.actions.setReplyData({replyData: value}));
    };
    const replyDataTextarea = replyDataContainer?.querySelector("textarea");
    if (replyDataTextarea) {
      replyDataTextarea.addEventListener("change", (e) => handleReplyDataOnChange(e.target.value));
    }

    const replyFileContainer = replyModal.querySelector(`[data-${inst.tag}-reply='replyDataFile']`);
    replyFileContainer.classList.add('hidden');
    const handleReplyFileOnChange = (target) => {
      const value = target.files.length ? target.files[0] : null;
      if (value) {
        inst.store.dispatch(inst.actions.setReplyData({replyDataFile: value, replyDataFileFormat: value.type}));
      }
    };
    const replyFileInput = replyFileContainer.querySelector("input");
    if (replyFileInput) {
      replyFileInput.addEventListener("change", (e) => handleReplyFileOnChange(e.target));
    }

    const handleReplyTypeOnChange = (e) => {
      if (!replyDataContainer.classList.contains('hidden')) {
        replyDataContainer.classList.add('hidden');
      }
      if (!replyFileContainer.classList.contains('hidden')) {
        replyFileContainer.classList.add('hidden');
      }
      const replyContainer = replyModal.querySelector(`[data-${inst.tag}-reply='${e.target.value}']`);
      if (replyContainer) {
        replyContainer.classList.remove('hidden');
      }
      inst.store.dispatch(inst.actions.clearReplyData());
    };
    const replyTypeSelect = replyModal.querySelector(`[data-${inst.tag}-reply='replyType']`).querySelector("select");
    replyTypeSelect.addEventListener("change", (e) => handleReplyTypeOnChange(e));

    const handleAcceptButtonOnClick = () => {
      if (!isAcceptOrRejectButtonActive()) {
        return;
      }
      let confirmation = confirm("Are you sure to reply accept?");
      if (confirmation) {
        inst.store.dispatch(inst.actions.replyMessage(messageId, 'ACCEPT'));
      }
    };
    const handleRejectButtonOnClick = () => {
      if (!isAcceptOrRejectButtonActive()) {
        return;
      }
      let confirmation = confirm("Are you sure to reply reject?");
      if (confirmation) {
        inst.store.dispatch(inst.actions.replyMessage(messageId, 'REJECT'));
      }
    };
    const handleReplyButtonOnClick = () => {
      if (!isAcceptOrRejectButtonActive()) {
        return;
      }
      let confirmation = confirm("Are you sure to reply?");
      if (confirmation) {
        inst.store.dispatch(inst.actions.replyMessage(messageId, null));
      }
    };
    const isAcceptOrRejectButtonActive = (replyType, value = "") => {
      const {replyData} = store.getState();
      let isActive;
      if (replyTypeSelect.value === 'replyData') {
        isActive = replyData && replyData["replyData"];
        if (!isActive) {
          alert("Please provide reply message")
        }
        isActive = isActive && checkInvalidCharacterInItem(replyData, ["replyData"]);
      } else {
        isActive = replyData && replyData["replyDataFile"];
        if (!isActive) {
          alert("Please provide reply data file")
        }
      }
      return isActive;
    };

    const acceptButton = replyModal.querySelector(`[data-${inst.tag}-reply='accept']`)?.querySelector("button");
    if (acceptButton) {
      const newAcceptButton = acceptButton.cloneNode(true);
      newAcceptButton.addEventListener("click", () => handleAcceptButtonOnClick());
      acceptButton.replaceWith(newAcceptButton); //Remove old event handler
    }

    const rejectButton = replyModal.querySelector(`[data-${inst.tag}-reply='reject']`)?.querySelector("button");
    if (rejectButton) {
      const newRejectButton = rejectButton.cloneNode(true);
      newRejectButton.addEventListener("click", () => handleRejectButtonOnClick());
      rejectButton.replaceWith(newRejectButton); //Remove old event handler
    }

    const replyButton = replyModal.querySelector(`[data-${inst.tag}-reply='reply']`)?.querySelector("button");
    if (replyButton) {
      const newReplyButton = replyButton.cloneNode(true);
      newReplyButton.addEventListener("click", () => handleReplyButtonOnClick());
      replyButton.replaceWith(newReplyButton); //Remove old event handler
    }
  };
  inst.buildFilter = () => {
    const queryParams = ['messageId', 'messageType', 'source', 'fileName', 'dcrReferenceId',
                         'status', 'createDatetimeFrom', 'createDatetimeTo'];
    const handleQueryInputOnChange = (name, target) => {
      let value;
      if (target.tagName === 'INPUT') {
        if (target.type === 'text') {
          value = target.value;
        } else if (target.type === 'checkbox') {
          value = target.checked;
        } else if (target.type === 'datetime-local' || target.type === 'date') {
          value = getFilterDateInputValue(name, target.value);
        }
      } else if (target.tagName === 'SELECT') {
        value = target.value;
      }
      const card = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='filter']`);
      const labelContainer = card.querySelector(`[data-${inst.tag}-filter='${name}']`);
      handleInputLabels(labelContainer, value, target.tagName);
      inst.store.dispatch(inst.actions.setQuery({[name]: value}));
    };
    const handleSearchButtonOnClick = () => {
      const {recordLoading} = inst.store.getState();
      if (recordLoading) {
        return;
      }
      if (!inst.isSearchButtonActive()) {
        return;
      }
      inst.store.dispatch(inst.actions.setPage());
    };
    const handleClearButtonOnClick = () => {
      const {recordLoading} = inst.store.getState();
      if (recordLoading) {
        return;
      }
      queryParams.forEach(q => {
        const fieldContainer = card.querySelector(`[data-${inst.tag}-filter='${q}']`);
        const inputs = Array.from(fieldContainer.querySelectorAll("input"));
        if (inputs && inputs.length) {
          inputs.forEach(input => {
            if (input.type === 'checkbox') {
              input.checked = false;
            } else {
              input.value = null;
            }
          });
        }
        const select = fieldContainer.querySelector("select");
        if (select) {
          select.selectedIndex = -1;
        }
      });
      const boxLabels = card.querySelectorAll("[data-dropdown='box']")
      boxLabels.forEach(elem => {
        elem.querySelector('p').innerHTML = "";
      });
      inst.store.dispatch(inst.actions.clearQuery());
      inst.store.dispatch(inst.actions.getRecords());
    };
    const buildCollapseCard = (card) => {
      const header = card.querySelector("[data-collapsecard='header']");
      header.addEventListener('click', () => handleCollapseCardHeaderOnClick(card));
    };
    const renderFilterContent = (card) => {
      queryParams.filter(q => q !== 'status' && q !== 'messageType').forEach(q => {
        const fieldContainer = card.querySelector(`[data-${inst.tag}-filter='${q}']`);
        const input = fieldContainer.querySelector("input");
        if (input) {
          input.addEventListener("change", (e) => handleQueryInputOnChange(q, e.target));
        }
        const select = fieldContainer.querySelector("select");
        if (select) {
          select.selectedIndex = -1;
          select.addEventListener("change", (e) => handleQueryInputOnChange(q, e.target));
        }
      });
      const statusSelect = card.querySelector(`[data-${inst.tag}-filter='status']`);
      statusSelect.querySelector("[data-dropdown='box']").addEventListener('click', () => handleDropDownOnClick(statusSelect));
      statusSelect.querySelector("[data-dropdown='mask']").addEventListener('click', () => handleDropDownOnClick(statusSelect));
      const statusInputs = statusSelect.querySelector("[data-dropdown='container']").querySelectorAll('input');
      const statusTarget = () => {
        return {
          value: Array.from(statusInputs).filter(i => i.checked).map(o => o.value),
          tagName: 'SELECT',
        };
      };
      statusInputs.forEach(input => {
        input.addEventListener('change', () => displayDropdownBoxLabel(statusSelect));
        input.addEventListener('change', () => handleQueryInputOnChange('status', statusTarget()));
      });

      const msgtypeSelect = card.querySelector(`[data-${inst.tag}-filter='messageType']`);
      msgtypeSelect.querySelector("[data-dropdown='box']").addEventListener('click', () => handleDropDownOnClick(msgtypeSelect));
      msgtypeSelect.querySelector("[data-dropdown='mask']").addEventListener('click', () => handleDropDownOnClick(msgtypeSelect));
      const msgtypeInputs = msgtypeSelect.querySelector("[data-dropdown='container']").querySelectorAll('input');
      const msgtypeTarget = () => {
        return {
          value: Array.from(msgtypeInputs).filter(i => i.checked).map(o => o.value),
          tagName: 'SELECT',
        };
      };
      msgtypeInputs.forEach(input => {
        input.addEventListener('change', () => displayDropdownBoxLabel(msgtypeSelect));
        input.addEventListener('change', () => handleQueryInputOnChange('messageType', msgtypeTarget()));
      });
    };
    inst.isSearchButtonActive = () => {
      const {query} = store.getState();
      const checkInvalidCharacterFields = ["messageId", "source", "fileName", "dcrReferenceId", "username", "messageRemark", "departmentCode"];
      let isActive = checkInvalidCharacterInItem(query, checkInvalidCharacterFields);
      return isActive;
    };
    const card = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='filter']`);
    buildCollapseCard(card);
    renderFilterContent(card);
    const searchButton = card.querySelector("#history-filter-search");
    searchButton.addEventListener("click", handleSearchButtonOnClick);
    const clearButton = card.querySelector("#history-filter-clear");
    clearButton.addEventListener("click", handleClearButtonOnClick);
  };
  inst.initialize = () => {
    const {tag} = inst.store.getState();
    inst.tag = tag;
    inst.buildFilter();
    inst.setTable();
    inst.setDataFileModal();
    inst.setDCRModal();
    inst.setPMDSCSTModal();
    inst.setPMDSWDLModal();
    store.dispatch(inst.actions.getRecords());
  };
  // Table related method
  return inst;
};
