"use strict";

import {store as oauth2Store, actions as oauth2Actions} from './model/accmgmt_oauth2_model.js';
import {store as signStore, actions as signActions} from './model/accmgmt_sign_model.js';
import {store as certStore, actions as certActions} from './model/accmgmt_cert_model.js';
import { store as decryptionStore, actions as decryptionActions } from './model/accmgmt_decryption_model.js';
import {getBaseAccmgmt} from './component/accmgmt_component.js';
import {convertDatetimeToLocaleDate, hasInvalidCharacter, onNavInitialize} from './common.js';
import {checkRequiredFieldInItem, checkInvalidCharacterInItem} from './common.js';

const OAuth2 = (store, actions) => {
  const base = getBaseAccmgmt('oauth2', store, actions);
  const inst = Object.create(base);
  inst.name = 'OAuth2';
  inst.renderViewItem = function(item, content) {
    const clientIdInput = content.querySelector("[data-accmgmt-content='clientId']").querySelector("input");
    clientIdInput.value = item.clientId ? item.clientId : null;
    const clientSecretInput = content.querySelector("[data-accmgmt-content='clientSecret']").querySelector("input");
    clientSecretInput.value = item.clientSecret ? item.clientSecret : null;
    const scopeInput = content.querySelector("[data-accmgmt-content='scope']").querySelector("input");
    scopeInput.value = item.scope ? item.scope : null;
  };
  inst.renderFormItem = function(content) {
    const clientIdInput = content.querySelector("[data-accmgmt-form='clientId']").querySelector("input");
    clientIdInput.addEventListener('change', (e) => this.handleFormInputOnChange('clientId', e.target.value));
    const clientSecretInput = content.querySelector("[data-accmgmt-form='clientSecret']").querySelector("input");
    clientSecretInput.addEventListener('change', (e) => this.handleFormInputOnChange('clientSecret', e.target.value));
    const scopeInput = content.querySelector("[data-accmgmt-form='scope']").querySelector("input");
    scopeInput.addEventListener('change', (e) => this.handleFormInputOnChange('scope', e.target.value));
  };
  inst.isFormSubmittable = function() {
    const {changeItem} = this.store.getState();
    let isValid = checkRequiredFieldInItem(changeItem, ['clientId', 'clientSecret']);
    isValid = isValid && checkInvalidCharacterInItem(changeItem, ['clientId', 'clientSecret']);
    return isValid;
  };
  return inst;
}


const Signing = (store, actions) => {
  const base = getBaseAccmgmt('sign', store, actions);
  const inst = Object.create(base);
  inst.name = 'Signing';
  inst.renderViewItem = function(item, content) {
    const signSecretInput = content.querySelector("[data-accmgmt-content='signingSecret']").querySelector("input");
    signSecretInput.value = item.signingSecret ? item.signingSecret : null;
  };
  inst.renderFormItem = function(content) {
    const signSecretInput = content.querySelector("[data-accmgmt-form='signingSecret']").querySelector("input");
    signSecretInput.addEventListener('change', (e) => this.handleFormInputOnChange('signingSecret', e.target.value));
  };
  inst.isFormSubmittable = function() {
    const {changeItem} = this.store.getState();
    let isValid = checkRequiredFieldInItem(changeItem, ['signingSecret']);
    isValid = isValid && checkInvalidCharacterInItem(changeItem, ['signingSecret']);
    return isValid;
  };
  return inst;
}

const Cert = (store, actions) => {
  const base = getBaseAccmgmt('cert', store, actions); 
  const inst = Object.create(base);
  inst.name = 'Cert';
  inst.store = store;
  inst.actions = actions;
  inst.store.subscribe((action) => {
    if (action.type === 'SET_QUERY_CERT_TYPE') {
      inst.store.dispatch(inst.actions.getItems());
    }
  });
  inst.buildCollapseCard = function() {
    this.buildCollapseCard = base.buildCollapseCard;
    this.buildCollapseCard();
    const card = document.getElementById(`accmgmt-${this.type}`);
    const tabs = card.querySelectorAll("[data-collapsecard='tab']");
    tabs.forEach(t => {
      t.addEventListener('click', () => this.handleQueryCertTypeOnChange(t.getAttribute('data-accmgmt-content')));
    });
    tabs[0].disabled = true;
  };
  inst.renderViewItem = function(item, content) {
    const commonNameInput = content.querySelector("[data-accmgmt-content='commonName']").querySelector("input");
    commonNameInput.value = item.commonName ? item.commonName : null;
    const orgNameInput = content.querySelector("[data-accmgmt-content='orgName']").querySelector("input");
    orgNameInput.value = item.orgName ? item.orgName : null;
    const issuerInput = content.querySelector("[data-accmgmt-content='issuer']").querySelector("input");
    issuerInput.value = item.issuer ? item.issuer : null;
    const serialNumberInput = content.querySelector("[data-accmgmt-content='serialNumber']").querySelector("input");
    serialNumberInput.value = item.serialNumber ? item.serialNumber : null;
    const validFromInput = content.querySelector("[data-accmgmt-content='validFrom']").querySelector("input");
    validFromInput.value = item.validFrom ? convertDatetimeToLocaleDate(item.validFrom) : null;
    const validUntilInput = content.querySelector("[data-accmgmt-content='validUntil']").querySelector("input");
    validUntilInput.value = item.validUntil ? convertDatetimeToLocaleDate(item.validUntil) : null;
  };
  inst.renderFormItem = function(content) {
    const prvKey = content.querySelector("[data-accmgmt-form='privateKey']");
    prvKey.classList.add('hidden');
    const prvKeyInput = prvKey.querySelector('input');
    prvKeyInput.addEventListener('change', (e) => this.handleFormInputOnChange('privateKey', e.target.files.length ? e.target.files[0] : null));
    const pubCert = content.querySelector("[data-accmgmt-form='publicCertificate']");
    const pubCertInput = pubCert.querySelector('input');
    pubCertInput.addEventListener('change', (e) => this.handleFormInputOnChange('publicCertificate', e.target.files.length ? e.target.files[0] : null));
    const certTypeSelect = content.querySelector("[data-accmgmt-form='certType']").querySelector('select');
    certTypeSelect.addEventListener('change', (e) => this.handleCertTypeSelectOnChange(prvKey, e.target.value));
    this.handleFormInputOnChange('certType', 'CRP_SERVER_CERTIFICATE');
  };
  inst.handleCertTypeSelectOnChange = function(prvKey, value) {
    const prvKeyInput = prvKey.querySelector('input');
    this.handleFormInputOnChange('certType', value);
    if (value === 'CRP_SERVER_CERTIFICATE') {
      prvKey.classList.add('hidden');
      prvKeyInput.disabled = true;
      prvKeyInput.value = null;
      this.handleFormInputOnChange('privateKey', null);
    } else {
      prvKey.classList.remove('hidden');
      prvKeyInput.disabled = false;
    }
  };
  inst.handleQueryCertTypeOnChange = function(value) {
    this.store.dispatch(this.actions.setQueryCertType(value));
    const card = document.getElementById(`accmgmt-${this.type}`);
    const tabs = card.querySelectorAll("[data-collapsecard='tab']");
    tabs.forEach(t => {
      if (t.getAttribute('data-accmgmt-content') === value) {
        t.disabled = true;
      } else {
        t.disabled = false;
      }
    });
  };
  inst.isFormSubmittable = function() {
    const {changeItem} = this.store.getState();
    let requiredFields;
    if (changeItem.certType === 'CRP_SERVER_CERTIFICATE') {
      requiredFields = ['certType', 'publicCertificate'];
    } else {
      requiredFields = ['certType', 'publicCertificate', 'privateKey'];
    }
    return checkRequiredFieldInItem(changeItem, requiredFields);
  };
  return inst;
};

const Decryption = (store, actions) => {
  const base = getBaseAccmgmt('decryption', store, actions);
  const inst = Object.create(base);
  inst.name = 'Decryption';
  inst.initialize = function() {
    this.store.subscribe((action) => {
      if (action.type === 'SET_DECRYPTED_DATA') {
        const {decryptedData} = this.store.getState();
        this.buildContent(atob(decryptedData));
      }
    });
    this.buildCollapseCard();
    this.buildContent();
  };
  inst.renderViewItem = function(decryptedData, content) {
    const encryptedDataInput = content.querySelector("[data-accmgmt-content='encryptedData']").querySelector("input");
    encryptedDataInput.addEventListener('change', (e) => {store.dispatch(actions.setEncryptedData(e.target.value));});
    
    const encryptedSymmetricKeyInput = content.querySelector("[data-accmgmt-content='encryptedSymmetricKey']").querySelector("input");
    encryptedSymmetricKeyInput.addEventListener('change', (e) => {store.dispatch(actions.setEncryptedSymmetricKey(e.target.value));});

    const decrypt = content.querySelector("[data-accmgmt-content='decrypt']");
    const decryptButton = decrypt.querySelector("button");
    decryptButton.addEventListener('click', () => {store.dispatch(actions.getDecryptedData());});

    const decryptedDataContainer = content.querySelector("[data-accmgmt-content='decryptedData']");
    const decryptedDataTextarea = decryptedDataContainer.querySelector("textarea");
    if (!!decryptedData) {
      decryptedDataContainer.classList.remove("hidden");
      decryptedDataTextarea.value = !!decryptedData ? decryptedData : null;
    } else {
      decryptedDataContainer.classList.add("hidden");
      decryptedDataTextarea.value = null;
    }
    const downloadButton = decryptedDataContainer.querySelector("[data-accmgmt-content='downloadButton']");
    downloadButton.addEventListener('click', () => {store.dispatch(actions.downloadDecryptedData());});
  };
  inst.buildContent = function(decryptedData) {
    // Overwrite the base method to handle decryption items
    const card = document.getElementById(`accmgmt-${this.type}`);
    const contentContainer = card.querySelector("[data-collapsecard='content-container']");
    contentContainer.innerHTML = "";

    const contentTemplate = document.getElementById(`accmgmt-${this.type}-content-template`).cloneNode(true);
    this.renderViewItem(decryptedData, contentTemplate.content);
    contentContainer.appendChild(contentTemplate.content);
  };
  return inst;
}



const oauth2 = OAuth2(oauth2Store, oauth2Actions);
const signing = Signing(signStore, signActions);
const cert = Cert(certStore, certActions);
const decryption = Decryption(decryptionStore, decryptionActions);

document.addEventListener("DOMContentLoaded", function(){
  onNavInitialize();

  oauth2.initialize();
  signing.initialize();
  cert.initialize();
  decryption.initialize();
});
