"use strict";

import {createStore, displayActionResponse, handleDownloadResponse} from '../common.js';
import {APIPrefix} from '../common.js';
import {convertUrlQueryParams} from '../common.js';

// Redux architecture pieces
export const actions = {
  // Actions for History Page
  setPage: (page) => {
    return {
      type: 'SET_PAGE',
      payload: page
    };
  },
  setSize: (size) => {
    return {
      type: 'SET_SIZE',
      payload: size
    };
  },
  setTotalPages: (totalPages) => {
    return {
      type: 'SET_TOTAL_PAGES',
      payload: totalPages
    };
  },
  setTotalRecords: (totalRecords) => {
    return {
      type: 'SET_TOTAL_RECORDS',
      payload: totalRecords
    };
  },
  getRecords: () => {
    return {
      type: 'GET_RECORDS',
    };
  },
  setRecords: (records) => {
    return {
      type: 'SET_RECORDS',
      payload: records,
    };
  },
  setRecordLoading: (isLoading) => {
    return {
      type: 'SET_RECORD_LOADING',
      payload: isLoading,
    };
  },
  getFile: (messageId, filename) => {
    return {
      type: 'GET_FILE',
      payload: {messageId, filename},
    };
  },
  setQuery: (param) => {
    return {
      type: 'SET_QUERY',
      payload: param
    };
  },
  clearQuery: () => {
    return {
      type: 'CLEAR_QUERY',
    };
  },
  getDetail: (messageId, filename) => {
    return {
      type: 'GET_DETAIL',
      payload: {messageId, filename},
    };
  },
  setDetail: (detail) => {
    return {
      type: 'SET_DETAIL',
      payload: detail
    };
  },
  setDetailLoading: (isLoading) => {
    return {
      type: 'SET_DETAIL_LOADING',
      payload: isLoading,
    };
  },
};

const initialState = {
  tag: 'sysrpt',
  tab: 'history',
  page: 0,
  totalPages: null,
  size: 10,
  records: [],
  recordLoading: false,
  query: {},
  detail: null,
};

const systemReportReducer = (state = initialState, action) => {
  let reduce;
  switch (action.type) {
    case 'SET_PAGE':
      reduce = (state, action) => {
        const {size, query} = state;
        let page;
        if (action.payload === null || action.payload === undefined) {
          page = initialState.page;
        } else {
          page = action.payload;
        }
        getSystemReportHistory(page, size, query);
        return Object.assign(Object.assign({}, state), {
          page: page,
        });
      };
      return reduce(state, action);

    case 'SET_SIZE':
      reduce = (state, action) => {
        const { query } = state;
        const newPage = 0;
        let newSize;
        if (action.payload === null || action.payload === undefined) {
          newSize = initialState.size;
        } else {
          newSize = action.payload;
        }
        getSystemReportHistory(newPage, newSize, query);
        return Object.assign(Object.assign({}, state), {
          page: newPage,
          size: newSize,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_PAGES':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalPages: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalRecords: action.payload,
        });
      }
      return reduce(state, action);

    case 'GET_RECORDS':
      // TODO: handle query param
      reduce = (state, action) => {
        const {page, size, query} = state;
        getSystemReportHistory(page, size, query);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          records: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_RECORD_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          recordLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_QUERY':
      reduce = (state, action) => {
        const queryItem = Object.assign({}, state.query);
        return Object.assign(Object.assign({}, state), {
          query: Object.assign(queryItem, action.payload),
        });
      };
      return reduce(state, action);

    case 'CLEAR_QUERY':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {query: {}, page: initialState.page});
      };
      return reduce(state, action);

    case 'GET_FILE':
      reduce = (state, action) => {
        downloadSystemReportFile(action.payload.messageId, action.payload.filename);
      return store.getState();
      };
      return reduce(state, action);

    case 'GET_DETAIL':
      reduce = (state, action) => {
        previewSystemReportFile(action.payload.messageId, action.payload.filename);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_DETAIL':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detail: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_DETAIL_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detailLoading: action.payload,
        });
      };
      return reduce(state, action);

    default:
      return state;
  }
};

export const store = createStore(systemReportReducer);

export const getNameFromMessageType = (messageType) => {
  const Names = {
    'rpt.dnl.nti.01': 'rptdnlnti',
  };
  return Names[messageType] ? Names[messageType] : null;
};

const getSystemReportHistory = (page, size, params = {}) => {
  store.dispatch(actions.setRecordLoading(true));
  const url = convertUrlQueryParams(`${APIPrefix}/crp-service/crp-system-report/history`, Object.assign({page, size}, params));
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    const {tab} = store.getState();
    if (tab !== 'history') {
      return;
    }
    if (result.result === "Success") {
      store.dispatch(actions.setRecords(result.data.record ? result.data.record : []));
      store.dispatch(actions.setTotalPages(result.data.totalPages));
      store.dispatch(actions.setTotalRecords(result.data.total));
      if (result.data && result.data.total === 0) {
        alert('No matched record found');
      }
    } else {
      displayActionResponse(result);
    }
    store.dispatch(actions.setRecordLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const downloadSystemReportFile = (messageId, filename) => {
  const url = `${APIPrefix}/crp-service/crp-system-report/${messageId}/file`;
  fetch(url).then(response => {
    handleDownloadResponse(response, filename);
  }).catch(error => {
    console.error(error);
  });

};


const previewSystemReportFile = (messageId, filename) => {
  store.dispatch(actions.setDetailLoading(true));
  const url = `${APIPrefix}/crp-service/crp-system-report/${messageId}/file`;
  fetch(url).then(response => {
    if (response.ok) {
      return response.blob().then(blob => {
        const pattern = /\.[0-9a-z]+$/;
        let fileExtMatch;
        if (blob.name) {
          fileExtMatch = blob.name.match(pattern);
        } else if (filename) {
          fileExtMatch = filename.match(pattern);
        }
        const fileExt = fileExtMatch ? fileExtMatch[0] : '';
        if (blob.type === 'application/json' || fileExt === 'json') {
          blob.text().then(data => {
            const detail = JSON.stringify(JSON.parse(data), null, '    ');
            store.dispatch(actions.setDetail(detail));
            store.dispatch(actions.setDetailLoading(false));
          }).catch(err => {
            store.dispatch(actions.setDetail(null));
            store.dispatch(actions.setDetailLoading(false));
          });
        } else if (blob.type === 'application/octet-stream') {
          blob.text().then(detail => {
            store.dispatch(actions.setDetail(detail));
            store.dispatch(actions.setDetailLoading(false));
          }).catch(err => {
            store.dispatch(actions.setDetail(null));
            store.dispatch(actions.setDetailLoading(false));
          });
        } else {
          store.dispatch(actions.setDetail(null));
          store.dispatch(actions.setDetailLoading(false));
        }
      });
    } else if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.text().then(text => {
        alert(`${text}`);
      });
    }
  }).catch(error => {
    console.error(error);
  });

};
