"use strict";

import {createStore, displayActionResponse} from '../common.js';
import {APIPrefix, getJsonRequestHeaderWithCsrf, convertUrlQueryParams} from '../common.js';
import {handleDownloadResponse} from '../common.js';

// Redux architecture pieces
export const actions = {
  // Actions for Data submission common
  setTab: (tab) => {
    return {
      type: 'SET_TAB',
      payload: tab,
    };
  },
  setSize: (size) => {
    return {
      type: 'SET_SIZE',
      payload: size
    };
  },
  // Actions for History Page
  setPage: (page) => {
    return {
      type: 'SET_PAGE',
      payload: page
    };
  },
  setTotalPages: (totalPages) => {
    return {
      type: 'SET_TOTAL_PAGES',
      payload: totalPages
    };
  },
  setTotalRecords: (totalRecords) => {
    return {
      type: 'SET_TOTAL_RECORDS',
      payload: totalRecords
    };
  },
  getRecords: () => {
    return {
      type: 'GET_RECORDS',
    };
  },
  setRecords: (records) => {
    return {
      type: 'SET_RECORDS',
      payload: records,
    };
  },
  setRecordLoading: (isLoading) => {
    return {
      type: 'SET_RECORD_LOADING',
      payload: isLoading,
    };
  },
  getDetail: (idType, id) => {
    return {
      type: 'GET_DETAIL',
      payload: {idType, id},
    };
  },
  setDetail: (detail) => {
    return {
      type: 'SET_DETAIL',
      payload: detail
    };
  },
  setDetailLoading: (isLoading) => {
    return {
      type: 'SET_DETAIL_LOADING',
      payload: isLoading,
    };
  },
  getFile: (idType, id, filename) => {
    return {
      type: 'GET_FILE',
      payload: {idType, id, filename},
    };
  },
  getArchive: (reportMessageId, filename) => {
    return {
      type: 'GET_ARCHIVE',
      payload: {reportMessageId, filename},
    };
  },
  getReport: (reportMessageId, filename) => {
    return {
      type: 'GET_REPORT',
      payload: {reportMessageId, filename},
    };
  },
  setQuery: (param) => {
    return {
      type: 'SET_QUERY',
      payload: param
    };
  },
  clearQuery: () => {
    return {
      type: 'CLEAR_QUERY',
    };
  },
  resolveMessage: (messageId) => {
    return {
      type: 'RESOLVE_MSG',
      payload: messageId
    };
  },
  setResolvedMessage: (resolvedMessage) => {
    return {
      type: 'SET_RESOLVE_MSG',
      payload: resolvedMessage
    };
  },
  clearResolvedMessage: () => {
    return {
      type: 'CLEAR_RESOLVE_MSG',
    };
  },
  // Actions for Enquiry Form
  setUploadItem: (item) => {
    return {
      type: 'SET_UPLOAD_ITEM',
      payload: item
    };
  },
  resetUploadItem: () => {
    return {
      type: 'RESET_UPLOAD_ITEM',
    };
  },
  createUploadItem: (tab) => {
    return {
      type: 'CREATE_UPLOAD_ITEM',
      payload: tab
    };
  },
};

const initialState = {
  tag: 'credrpt',
  tab: null,
  page: 0,
  totalPages: null,
  size: 10,
  records: [],
  recordLoading: false,
  detail: {},
  detailLoading: false,
  query: {},
  resolvedMessage: null,
  uploadItem: {},
};

const creditReportReducer = (state = initialState, action) => {
  let reduce, idType;
  switch (action.type) {
    case 'SET_TAB':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          tab: action.payload,
          uploadItem: {},
          query: {},
        });
      };
      return reduce(state, action);

    case 'SET_PAGE':
      reduce = (state, action) => {
        const { size, query } = state;
        let page;
        if (action.payload === null || action.payload === undefined) {
          page = initialState.page;
        } else {
          page = action.payload;
        }
        getCreditReportHistory(page, size, query);
        return Object.assign(Object.assign({}, state), {
          page: page,
        });
      };
      return reduce(state, action);

    case 'SET_SIZE':
      reduce = (state, action) => {
        const {query} = state;
        const newPage = 0;
        let newSize;
        if (action.payload === null || action.payload === undefined) {
          newSize = initialState.size;
        } else {
          newSize = action.payload;
        }
        getCreditReportHistory(newPage, newSize, query);
        return Object.assign(Object.assign({}, state), {
          page: newPage,
          size: newSize,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_PAGES':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalPages: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_RECORDS':
      reduce = (state, action) => {
          return Object.assign(Object.assign({}, state), {
              totalRecords: action.payload,
          });
      };
      return reduce(state, action);

    case 'GET_RECORDS':
      // TODO: handle query param
      reduce = (state, action) => {
        const {page, size, query} = state;
        getCreditReportHistory(page, size, query);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          records: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_RECORD_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          recordLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_QUERY':
      reduce = (state, action) => {
        const queryItem = Object.assign({}, state.query);
        return Object.assign(Object.assign({}, state), {
          query: Object.assign(queryItem, action.payload),
        });
      };
      return reduce(state, action);

    case 'CLEAR_QUERY':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {query: {}, page: initialState.page});
      };
      return reduce(state, action);

    case 'GET_DETAIL':
      getCreditReportDetail(action.payload.idType, action.payload.id);
      // getFakeCreditReportDetail(action.payload.idType, action.payload.id);
      return store.getState();

    case 'SET_DETAIL':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detail: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_DETAIL_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detailLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'GET_FILE':
      downloadFile(action.payload.idType, action.payload.id, action.payload.filename);
      return store.getState();

    case 'GET_ARCHIVE':
      downloadArchive(action.payload.reportMessageId, action.payload.filename);
      return store.getState();

    case 'GET_REPORT':
      downloadReport(action.payload.reportMessageId, action.payload.filename);
      return store.getState();

    case 'RESOLVE_MSG':
      const {resolvedMessage} = state;
      idType = 'messageId';
      resolveCreditReportMessage(idType, action.payload, resolvedMessage);
      return store.getState();

    case 'SET_RESOLVE_MSG':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          resolvedMessage: action.payload,
        });
      };
      return reduce(state, action);

    case 'CLEAR_RESOLVE_MSG':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {resolvedMessage: null});
      };
      return reduce(state, action);

    case 'SET_UPLOAD_ITEM':
      reduce = (state, action) => {
        const uploadItem = Object.assign({}, state.uploadItem);
        return Object.assign(Object.assign({}, state), {
          uploadItem: Object.assign(uploadItem, action.payload),
        });
      };
      return reduce(state, action);

    case 'RESET_UPLOAD_ITEM':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {uploadItem: {}});
      };
      return reduce(state, action);

    case 'CREATE_UPLOAD_ITEM':
      enquiryUpload(state.uploadItem);
      return store.getState();

    default:
      return state;
  }
};

export const store = createStore(creditReportReducer);

const getCreditReportHistory = (page, size, params = {}) => {
  store.dispatch(actions.setRecordLoading(true));
  const url = convertUrlQueryParams(`${APIPrefix}/crp-service/credit-report/history`, Object.assign({page, size}, params));
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    const {tab} = store.getState();
    if (tab !== 'history') {
      return;
    }
    if (result.result === "Success") {
      store.dispatch(actions.setRecords(result.data.record ? result.data.record : []));
      store.dispatch(actions.setTotalPages(result.data.totalPages));
      store.dispatch(actions.setTotalRecords(result.data.total));
      if (result.data && result.data.total === 0) {
        alert('No matched record found');
      }
    } else {
      displayActionResponse(result);
    }
    store.dispatch(actions.setRecordLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const enquiryUpload = (enquiryItem) => {                                            
  const {dataFile, fileFormat, messageRemark, departmentCode, targetedCRAs} = enquiryItem;
  let body = new FormData();
  body.append('dataFile', dataFile);
  body.append('fileFormat', fileFormat);
  body.append('targetedCRAs', targetedCRAs);
  if (messageRemark !== undefined && messageRemark !== null) {
    body.append('messageRemark', messageRemark);
  }
  if (departmentCode !== undefined && departmentCode !== null && departmentCode !== '') {
    body.append('recordCount', departmentCode);
  }
  fetch(`${APIPrefix}/crp-service/credit-report/enquiry/upload`, {
    method: 'POST',            
    headers: getJsonRequestHeaderWithCsrf(),
    body: body
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    if (result.result === "Success") {
      store.dispatch(actions.resetUploadItem());
      store.dispatch(actions.setTab('history'));
    }
  }).catch(error => {
    console.error(error);
  });
};

const getCreditReportDetail = (idType, messageId) => {
  GetCreditReportDetail(store, actions, idType, messageId);
};

export const GetCreditReportDetail = (curStore, curActions, idType, messageId) => {
  curStore.dispatch(curActions.setDetailLoading(true));
  const url = `${APIPrefix}/crp-service/credit-report/detail/${idType}/${messageId}`;
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      curStore.dispatch(curActions.setDetail(result.data));
    } else {
      displayActionResponse(result);
    }
    curStore.dispatch(curActions.setDetailLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const getFakeCreditReportDetail = (pmdsType, idType, messageId) => {
  store.dispatch(actions.setDetailLoading(true));
  const ret ={"result":"Success","resp_code":"0000","resp_msg":null,"data":{"id":18,"messageId":"M1234A20220803121900bd56f8705027","actionType":"CREDIT_REPORT_ENQUIRY","channel":"PORTAL","user":"super_admin","departmentCode":"0","filename":"FCENQ1234A20220803121900bae1c59968e","fileVersion":null,"fileSize":512,"fileAction":null,"recordCount":2,"targetedCRAs":["CRATU"],"createDatetime":"2022-08-03T16:18:09.080+00:00","status":"SUBMITTED","messageRemark":"test4Aug0017","crpResponse":{"id":null,"messageId":"MCRP20220803FD4B0","returnCode":"CRP00000","returnDescription":"Success","fileName":null,"fileVersion":null},"craResponses":[{"id":null,"source":"CRA01","messageId":"MCRP20220803FF14E","message":"","dataFileName":null,"dataMimeType":"text/plain","fileName":null,"fileVersion":null}],"creditReports":[{"generatedCRA":"CRATU","messageId":"MCRP20220803045606ecf01d113c631","reportFileName":"dummy.zip","innerFileNames":["uniqueFileName.meta","uniqueFileName.meta","uniqueFileName.meta","uniqueFileName.meta"]},{"generatedCRA":"CRA001","messageId":"MCRP20220803045606ecf01d113c631","reportFileName":"dummy.zip","innerFileNames":["uniqueFileName.meta"]}],"resolved":false,"resolvedMessage":null}} ;
  setTimeout(() => {
    store.dispatch(actions.setDetail(ret.data));
    store.dispatch(actions.setDetailLoading(false));
  });
};


const resolveCreditReportMessage = (idType, messageId, resolvedMessage) => {
  const body = {resolvedMessage};
  fetch(`${APIPrefix}/crp-service/credit-report/detail/${idType}/${messageId}`, {
    method: 'PUT',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    if (result.result === "Success") {
      store.dispatch(actions.clearResolvedMessage());
      store.dispatch(actions.getDetail(idType, messageId));
      store.dispatch(actions.getRecords());
    }
  }).catch(error => {
    console.error(error);
  });
};

const downloadFile = (idType, messageId, filename) => {
  const url = `${APIPrefix}/crp-service/credit-report/detail/${idType}/${messageId}/file`;
  fetch(url).then(response => {
    handleDownloadResponse(response, filename);
  }).catch(error => {
    console.error(error);
  });
};

const downloadArchive = (reportMessageId, filename) => {
  const url = `${APIPrefix}/crp-service/credit-report/detail/messageId/${reportMessageId}/report`;
  fetch(url).then(response => {
    handleDownloadResponse(response, filename, 'zip');
  }).catch(error => {
    console.error(error);
  });
};

const downloadReport = (reportMessageId, filename) => {
  const url = `${APIPrefix}/crp-service/credit-report/detail/messageId/${reportMessageId}/report/${filename}`;
  fetch(url).then(response => {
    handleDownloadResponse(response, filename);
  }).catch(error => {
    console.error(error);
  });
};
