"use strict";

import {createStore, displayActionResponse, handleDownloadResponse} from '../common.js';
import {APIPrefix, getRequestHeaderWithCsrf, getJsonRequestHeaderWithCsrf} from '../common.js';
import {convertUrlQueryParams} from '../common.js';

// Redux architecture pieces
export const actions = {
  // Actions for History Page
  setPage: (page) => {
    return {
      type: 'SET_PAGE',
      payload: page
    };
  },
  setSize: (size) => {
    return {
      type: 'SET_SIZE',
      payload: size
    };
  },
  setTotalPages: (totalPages) => {
    return {
      type: 'SET_TOTAL_PAGES',
      payload: totalPages
    };
  },
  setTotalRecords: (totalRecords) => {
    return {
      type: 'SET_TOTAL_RECORDS',
      payload: totalRecords
    };
  },
  getRecords: () => {
    return {
      type: 'GET_RECORDS',
    };
  },
  setRecords: (records) => {
    return {
      type: 'SET_RECORDS',
      payload: records,
    };
  },
  setRecordLoading: (isLoading) => {
    return {
      type: 'SET_RECORD_LOADING',
      payload: isLoading,
    };
  },
  getDetail: (messageType, messageId) => {
    return {
      type: 'GET_DETAIL',
      payload: {messageType, messageId},
    };
  },
  setDetail: (detail) => {
    return {
      type: 'SET_DETAIL',
      payload: detail
    };
  },
  setDetailLoading: (isLoading) => {
    return {
      type: 'SET_DETAIL_LOADING',
      payload: isLoading,
    };
  },
  getFile: (messageId, filename) => {
    return {
      type: 'GET_FILE',
      payload: {messageId, filename},
    };
  },
  setQuery: (param) => {
    return {
      type: 'SET_QUERY',
      payload: param
    };
  },
  clearQuery: () => {
    return {
      type: 'CLEAR_QUERY',
    };
  },
  replyMessage: (messageId, result) => {
    return {
      type: 'REPLY_MSG',
      payload: {messageId, result},
    };
  },
  setReplyData: (replyData) => {
    return {
      type: 'SET_REPLY_DATA',
      payload: replyData,
    };
  },
  setReplyLoading: (isLoading) => {
    return {
      type: 'SET_REPLY_LOADING',
      payload: isLoading,
    };
  },
  clearReplyData: () => {
    return {
      type: 'CLEAR_REPLY_DATA',
    };
  },
  // resolveMessage: (messageId) => {
  //   return {
  //     type: 'RESOLVE_MSG',
  //     payload: messageId
  //   };
  // },
  // setResolvedMessage: (resolvedMessage) => {
  //   return {
  //     type: 'SET_RESOLVE_MSG',
  //     payload: resolvedMessage
  //   };
  // },
  // clearResolvedMessage: () => {
  //   return {
  //     type: 'CLEAR_RESOLVE_MSG',
  //   };
  // },
};

const initialState = {
  tag: 'business',
  tab: 'history',
  page: 0,
  totalPages: null,
  size: 10,
  records: [],
  recordLoading: false,
  detail: {},
  detailType: null,
  detailLoading: false,
  resolvedMessage: null,
  replyData: {},
  replyLoading: false,
  query: {},
};

export const getDetailTypeFromMessageType = (messageType) => {
  const DetailTypes = {
    'data-file': 'data-file',
    'dcr': 'dcr',
    'dtsb.dnl.nti.01': 'data-file',
    'dtcr.dnl.nti.01': 'dcr',
    'pmds-withdrawal-prescribed-consent-list': 'pmds-withdrawal-prescribed-consent-list',
    'pmds-prescribed-consent-list': 'pmds-prescribed-consent-list',
    'pmds.cstdnl.nti.01': 'pmds-prescribed-consent-list',
    'pmds.wdldnl.nti.01': 'pmds-withdrawal-prescribed-consent-list',
  };
  return DetailTypes[messageType] ? DetailTypes[messageType] : null;
};

export const getMessageTypeNameFromMessageType = (messageType) => {
  const DetailTypes = {
    'dtsb.dnl.nti.01': '',
    'dtcr.dnl.nti.01': 'dcr',
  };
  return DetailTypes[messageType] ? DetailTypes[messageType] : null;
};

const businessNotificationReducer = (state = initialState, action) => {
  let reduce;
  switch (action.type) {
    case 'SET_PAGE':
      reduce = (state, action) => {
        const {size, query} = state;
        let page;
        if (action.payload === null || action.payload === undefined) {
          page = initialState.page;
        } else {
          page = action.payload;
        }
        getBusinessHistory(page, size, query);
        return Object.assign(Object.assign({}, state), {
          page: page,
        });
      };
      return reduce(state, action);

    case 'SET_SIZE':
      reduce = (state, action) => {
        const { query } = state;
        const newPage = 0;
        let newSize;
        if (action.payload === null || action.payload === undefined) {
          newSize = initialState.size;
        } else {
          newSize = action.payload;
        }
        getBusinessHistory(newPage, newSize, query);
        return Object.assign(Object.assign({}, state), {
          page: newPage,
          size: newSize,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_PAGES':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalPages: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_TOTAL_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          totalRecords: action.payload,
        });
      };
      return reduce(state, action);

    case 'GET_RECORDS':
      // TODO: handle query param
      reduce = (state, action) => {
        const {page, size, query} = state;
        getBusinessHistory(page, size, query);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_RECORDS':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          records: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_RECORD_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          recordLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_QUERY':
      reduce = (state, action) => {
        const queryItem = Object.assign({}, state.query);
        return Object.assign(Object.assign({}, state), {
          query: Object.assign(queryItem, action.payload),
        });
      };
      return reduce(state, action);

    case 'CLEAR_QUERY':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {query: {}, page: initialState.page});
      };
      return reduce(state, action);

    case 'GET_DETAIL':
      getBusinessDetail(getDetailTypeFromMessageType(action.payload.messageType), action.payload.messageId);
      return store.getState();

    case 'SET_DETAIL':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detail: action.payload,
          detailType: getDetailTypeFromMessageType(action.payload.messageType),
        });
      };
      return reduce(state, action);

    case 'SET_DETAIL_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          detailLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'GET_FILE':
      reduce = (state, action) => {
        const {detailType} = state;
        downloadBusinessFile(detailType, action.payload.messageId, action.payload.filename);
        return store.getState();
      };
      return reduce(state, action);

    case 'RESOLVE_MSG':
      reduce = (state, action) => {
        const {resolvedMessage, detailType} = state;
        resolveBusinessMessage(detailType, action.payload, resolvedMessage);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_RESOLVE_MSG':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          resolvedMessage: action.payload,
        });
      };
      return reduce(state, action);

    case 'CLEAR_RESOLVE_MSG':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {resolvedMessage: null});
      };
      return reduce(state, action);

    case 'REPLY_MSG':
      reduce = (state, action) => {
        const {replyData, detailType} = state;
        replyDCRNotification(action.payload.messageId, action.payload.result, replyData, detailType);
        return store.getState();
      };
      return reduce(state, action);

    case 'SET_REPLY_LOADING':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          replyLoading: action.payload,
        });
      };
      return reduce(state, action);

    case 'SET_REPLY_DATA':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {
          replyData: action.payload,
        });
      };
      return reduce(state, action);

    case 'CLEAR_REPLY_DATA':
      reduce = (state, action) => {
        return Object.assign(Object.assign({}, state), {replyData: {}});
      };
      return reduce(state, action);

    default:
      return state;
  }
};

export const store = createStore(businessNotificationReducer);

const getBusinessHistory = (page, size, params = {}) => {
  store.dispatch(actions.setRecordLoading(true));
  const url = convertUrlQueryParams(`${APIPrefix}/notification-centre/business/history`, Object.assign({page, size}, params));
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    const {tab} = store.getState();
    if (tab !== 'history') {
      return;
    }
    if (result.result === "Success") {
      store.dispatch(actions.setRecords(result.data.record ? result.data.record : []));
      store.dispatch(actions.setTotalPages(result.data.totalPages));
      store.dispatch(actions.setTotalRecords(result.data.total));
      if (result.data && result.data.total === 0) {
        alert('No matched record found');
      }
    } else {
      displayActionResponse(result);
    }
    store.dispatch(actions.setRecordLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const getBusinessDetail = (detailType, messageId) => {
  GetBusinessDetail(store, actions, detailType, messageId);
};

export const GetBusinessDetail = (curStore, curActions, detailType, messageId) => {
  curStore.dispatch(curActions.setDetailLoading(true));
  const url = `${APIPrefix}/notification-centre/business/${detailType}-download-noti/${messageId}`;
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    const {tab} = curStore.getState();
    console.log(tab);
    if (tab !== 'history' && tab !== 'request') {
      return;
    }
    if (result.result === "Success") {
      if (curActions.clearReplyData) {
        curStore.dispatch(curActions.clearReplyData());
      }
      curStore.dispatch(curActions.setDetail(result.data));
    } else {
      displayActionResponse(result);
    }
    curStore.dispatch(curActions.setDetailLoading(false));
  }).catch(error => {
    console.error(error);
  });
};

const downloadBusinessFile = (detailType, messageId, filename) => {
  const url = `${APIPrefix}/notification-centre/business/${detailType}-download-noti/${messageId}/file`;
  fetch(url).then(response => {
    handleDownloadResponse(response, filename);
  }).catch(error => {
    console.error(error);
  });

};

const resolveBusinessMessage = (detailType, messageId, resolvedMessage) => {
  const body = {resolvedMessage};
  fetch(`${APIPrefix}/notification-centre/business/${detailType}-download-noti/${messageId}`, {
    method: 'PUT',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    if (result.result === "Success") {
      store.dispatch(actions.clearResolvedMessage());
    }
    store.dispatch(actions.getRecords());
    store.dispatch(actions.getDetail(detailType, messageId));
  }).catch(error => {
    console.error(error);
  });
};

const replyDCRNotification = (messageId, result, replyData, detailType = 'dcr') => {
  store.dispatch(actions.setReplyLoading(true));
  const formData = new FormData();
  if (result) {
    formData.append('result', result);
  }
  Object.keys(replyData).forEach(k => {
    if (replyData[k]) {
      formData.append(k, replyData[k]);
    }
  });
  fetch(`${APIPrefix}/notification-centre/business/${detailType}-download-noti/${messageId}/reply`, {
    method: 'POST',
    headers: getRequestHeaderWithCsrf(),
    body: formData,
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    if (result.result === "Success") {
      store.dispatch(actions.clearReplyData());
      store.dispatch(actions.getRecords());
      store.dispatch(actions.getDetail(detailType, messageId));
    }
    store.dispatch(actions.setReplyLoading(false));
  }).catch(error => {
    console.error(error);
  });
};
