"use strict";

import {createStore, displayActionResponse} from '../common.js';
import {APIPrefix, getJsonRequestHeaderWithCsrf} from '../common.js';

// Redux architecture pieces
export const actions = {
  getItems: () => {
    // Get items from fetch
    return {
      type: 'GET_ITEMS'
    };
  },
  setItems: (items) => {
    // To be called by getItems callback
    return {
      type: 'SET_ITEMS',
      payload: items,
    };
  },
  setChangeItem: (item) => {
    return {
      type: 'SET_CHANGE_ITEM',
      payload: item,
    };
  },
  resetChangeItem: () => {
    return {
      type: 'RESET_CHANGE_ITEM',
    };
  },
  updateItem: (id, status) => {
    return {
      type: 'UPDATE_ITEM',
      payload: {
        id,
        status,
      }
    };
  },
  deleteItem: (id) => {
    return {
      type: 'DELETE_ITEM',
      payload: id,
    };
  },
  createItem: () => {
    return {
      type: 'CREATE_ITEM',
    };
  },
};

const initialState = {
  items: [],
  changeItem: {},
};

const accmgmtOauth2Reducer = (state = initialState, action) => {
  let changeItem;
  switch (action.type) {
    case 'GET_ITEMS':
      getOauth2Items();
      return state;

    case 'SET_ITEMS':
      // Set items & initialize page + currentPageItems
      return Object.assign(Object.assign({}, state), {
        items: action.payload,
      });

    case 'SET_CHANGE_ITEM':
      changeItem = Object.assign(
         Object.assign({}, state.changeItem), action.payload);
      return Object.assign(Object.assign({}, state), {changeItem});

    case 'RESET_CHANGE_ITEM':
      changeItem = Object.assign({}, initialState.changeItem);
      return Object.assign(Object.assign({}, state), {changeItem});

    case 'UPDATE_ITEM':
      updateOauth2(action.payload.id, action.payload.status);
      return state;

    case 'DELETE_ITEM':
      deleteOauth2(action.payload);
      return state;

    case 'CREATE_ITEM':
      postOauth2(state.changeItem);
      return state;

    default:
      return state;
  }
};

export const store = createStore(accmgmtOauth2Reducer);

const getOauth2Items = (params) => {
  const url = `${APIPrefix}/secret/oauth2`;
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      store.dispatch(actions.setItems(result.data));
    } else {
      const resultStatus = result && result.result ? result.result : 'Error:';
      const errMsg = result && result.resp_msg ?
      result.resp_msg : (result && result.error ? result.error : '');
      alert(`${resultStatus} ${errMsg}`);
    }
  }).catch(error => {
    console.error(error);
  });
};

const postOauth2 = (changeItem) => {                                            
  const body = Object.assign({}, changeItem);
  fetch(`${APIPrefix}/secret/oauth2`, {
    method: 'POST',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const deleteOauth2 = (id) => {
  fetch(`${APIPrefix}/secret/oauth2/${id}`, {
    method: 'DELETE',
    headers: getJsonRequestHeaderWithCsrf(),
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const updateOauth2 = (id, status) => {
  const body = Object.assign({}, {status});
  fetch(`${APIPrefix}/secret/oauth2/${id}`, {
    method: 'PUT',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};
