"use strict";

import {createStore, convertUrlQueryParams, displayActionResponse} from '../common.js';
import {APIPrefix, getJsonRequestHeaderWithCsrf} from '../common.js';

// Redux architecture pieces
export const actions = {
  getItems: () => {
    // Get items from fetch
    return {
      type: 'GET_ITEMS'
    };
  },
  setItems: (items) => {
    // To be called by getItems callback
    return {
      type: 'SET_ITEMS',
      payload: items,
    };
  },
  setQueryCertType: (certType) => {
    return {
      type: 'SET_QUERY_CERT_TYPE',
      payload: certType,
    };
  },
  setChangeItem: (item) => {
    return {
      type: 'SET_CHANGE_ITEM',
      payload: item,
    };
  },
  resetChangeItem: () => {
    return {
      type: 'RESET_CHANGE_ITEM',
    };
  },
  updateItem: (id, status) => {
    return {
      type: 'UPDATE_ITEM',
      payload: {
        id,
        status,
      }
    };
  },
  deleteItem: (id) => {
    return {
      type: 'DELETE_ITEM',
      payload: id,
    };
  },
  createItem: () => {
    return {
      type: 'CREATE_ITEM',
    };
  },
};

const initialState = {
  items: [],
  queryCertType: 'CRP_SERVER_CERTIFICATE',
  changeItem: {},
};

const accmgmtCertificateReducer = (state = initialState, action) => {
  let changeItem;
  switch (action.type) {
    case 'GET_ITEMS':
      getCertItems(state.queryCertType);
      return state;

    case 'SET_ITEMS':
      // Set items & initialize page + currentPageItems
      return Object.assign(Object.assign({}, state), {
        items: action.payload,
      });

    case 'SET_QUERY_CERT_TYPE':
      return Object.assign(Object.assign({}, state), {
        queryCertType: action.payload,
      });

    case 'SET_CHANGE_ITEM':
      changeItem = Object.assign(
         Object.assign({}, state.changeItem), action.payload);
      return Object.assign(Object.assign({}, state), {changeItem});

    case 'RESET_CHANGE_ITEM':
      changeItem = Object.assign({}, initialState.changeItem);
      return Object.assign(Object.assign({}, state), {changeItem});

    case 'UPDATE_ITEM':
      updateCert(action.payload.id, action.payload.status);
      return state;

    case 'DELETE_ITEM':
      deleteCert(action.payload);
      return state;

    case 'CREATE_ITEM':
      postCert(state.changeItem);
      return state;

    default:
      return state;
  }
};

export const store = createStore(accmgmtCertificateReducer);

const getCertItems = (certType) => {
  const url = convertUrlQueryParams(`${APIPrefix}/secret/certificate`, {certType});
  fetch(url).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    if (result.result === "Success") {
      if (result && result.data && !result.data.length) {
        alert("No record found");
      }
      store.dispatch(actions.setItems(result.data));
    } else {
      displayActionResponse(result);
    }
  }).catch(error => {
    console.error(error);
  });
};

const postCert = (changeItem) => {                                            
  const {publicCertificate, privateKey, certType} = changeItem;
  let body = new FormData();
  body.append('publicCertificate', publicCertificate);
  body.append('privateKey', privateKey);
  if (certType) {
    body.append('certType', certType);
  }
  fetch(`${APIPrefix}/secret/certificate`, {
    method: 'POST',            
    headers: getJsonRequestHeaderWithCsrf(),
    body: body
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const deleteCert = (id) => {
  fetch(`${APIPrefix}/secret/certificate/${id}`, {
    method: 'DELETE',
    headers: getJsonRequestHeaderWithCsrf(),
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};

const updateCert = (id, status) => {
  const body = Object.assign({}, {status});
  fetch(`${APIPrefix}/secret/certificate/${id}`, {
    method: 'PUT',
    headers: getJsonRequestHeaderWithCsrf({
      'Content-Type': 'application/json'
    }),
    body: JSON.stringify(body)
  }).then(response => {
    if (response.redirected) {
      window.location.href = response.url;
    } else {
      return response.json();
    }
  }).then(result => {
    displayActionResponse(result);
    store.dispatch(actions.getItems());
  }).catch(error => {
    console.error(error);
  });
};
