"use strict";

import {store, actions, roleStatesToPermissionIds, getPermissionIdsToRoleNameMap} from './model/group_model.js';
import {Usermgmt} from './component/usermgmt_component.js';
import {hasInvalidCharacter, arrayIsEqual} from './common.js';
import {checkRequiredFieldInItem, checkInvalidCharacterInItem} from './common.js';
import {onNavInitialize} from './common.js';
import {camelCaseToSentenceCase} from './common.js';
import {buildRoleCheckboxes, HandleCheckboxOnClick, renderStates} from './component/roletree.js';
import {getReorderedHeadersAndRecArrs} from './component/history_export.js';

const usermgmt = Usermgmt(store, actions);

usermgmt.fillUsermgmtTableRow = (cols, item, viewButton) => {
  cols[0].textContent = item.groupName
  cols[1].appendChild(viewButton);
  cols[1].classList.add('text-right');
};

const handleCheckboxOnClick = HandleCheckboxOnClick(store, actions);

usermgmt.renderViewItemContent = (formTable, itm) => {
  const groupNameItem = formTable.querySelector("[data-usermgmt-detail='groupName']");
  const groupNameInput = groupNameItem.querySelector("input");
  groupNameInput.value = itm.groupName ? itm.groupName : null;
  groupNameInput.addEventListener('input', (e) => usermgmt.handleFormInputOnChange('groupName', e.target.value));

  const groupIdItem = formTable.querySelector("[data-usermgmt-detail='id']");
  const groupIdInput = groupIdItem.querySelector("input");
  groupIdInput.name = 'id';
  groupIdInput.value = itm.id ? itm.id : null;
  groupIdInput.disabled = true;
  
  const {roleTree} = store.getState();
  buildRoleCheckboxes(roleTree, handleCheckboxOnClick);
};

usermgmt.renderAddItemContent = (formTable) => {
  const groupNameItem = formTable.querySelector("[data-usermgmt-detail='groupName']");
  const groupNameInput = groupNameItem.querySelector("input");
  groupNameInput.name = 'groupName';
  groupNameInput.addEventListener('input', (e) => usermgmt.handleFormInputOnChange('groupName', e.target.value));

  const groupIdItem = formTable.querySelector("[data-usermgmt-detail='id']");
  groupIdItem.remove();

  const {roleTree} = store.getState();
  buildRoleCheckboxes(roleTree, handleCheckboxOnClick);
};

Object.assign(usermgmt, {_handleViewButtonOnClick: usermgmt.handleViewButtonOnClick});
usermgmt.handleViewButtonOnClick = (itm) => {
  usermgmt._handleViewButtonOnClick(itm);
  store.dispatch(actions.initRoleStates(itm.id))
};
Object.assign(usermgmt, {_handleAddButtonOnClick: usermgmt.handleAddButtonOnClick});
usermgmt.handleAddButtonOnClick = () => {
  usermgmt._handleAddButtonOnClick();
  store.dispatch(actions.initRoleStates())
};

usermgmt.isAddItemButtonActive = () => {
  const {changeItem} = store.getState();
  let isValid = checkRequiredFieldInItem(changeItem, ['groupName']);
  isValid = isValid && checkInvalidCharacterInItem(changeItem, ['groupName']);
  return isValid;
};
usermgmt.isUpdateItemButtonActive = () => {
  const {changeItem, items, roleTree, roleStates} = store.getState();
  const item = items.find(itm => itm.id === changeItem.id);
  let isActive = false;
  if (item) {
    if (changeItem.groupName !== item.groupName) {
      isActive = true;
    };
    const permissionIds = roleStatesToPermissionIds(roleTree, roleStates);
    if (!arrayIsEqual(permissionIds.sort(), item.permissionIds.sort())) {
      isActive = true;
    }
  }
  if (!isActive) {
    alert("No updated field")
    return isActive;
  }
  isActive = isActive && checkInvalidCharacterInItem(changeItem, ['groupName']);
  return isActive;
};
usermgmt.isSearchButtonActive = () => {
  const {queryParams} = store.getState();
  return checkInvalidCharacterInItem(queryParams, ['groupName']);
};

usermgmt.exportRecords = () => {
  const {items, roleTree} = store.getState();
  // Deep Copy
  const records = JSON.parse(JSON.stringify(items));
  const roleNamesMap = getPermissionIdsToRoleNameMap(roleTree);
  records.forEach(rec => {
    const groupNames = rec.permissionIds.map(id => roleNamesMap[id]);
    rec.roleNames = groupNames.join('|');
  });
  const order = ['id', 'groupName', 'permissionIds', 'roleNames'];
  let {headers, recArrs} = getReorderedHeadersAndRecArrs(records, order)
  headers = headers.map(camelCaseToSentenceCase);
  return {filename: 'user_group_export.csv', headers, recArrs};
};

store.subscribe((action) => {
  if (action.type === 'SET_ROLETREE') {
    const {currentPage, itemPerPage, items, currentPageItems} = store.getState();
    usermgmt.buildUsermgmtLayout();
    usermgmt.buildUsermgmtTable(currentPageItems);
    usermgmt.buildUsermgmtPagination(currentPage, itemPerPage, items.length);
  } else if (action.type === 'SET_ITEMS' || action.type === 'SET_PAGE') {
    const {currentPage, itemPerPage, items, currentPageItems} = store.getState();
    usermgmt.buildUsermgmtTable(currentPageItems);
    usermgmt.buildUsermgmtPagination(currentPage, itemPerPage, items.length);
  } else if (action.type === 'GET_ROLE_STATES') {
    const {roleStates} = store.getState();
    renderStates(roleStates);
  } else if (action.type === 'INIT_ROLE_STATES') {
    const {roleStates} = store.getState();
    renderStates(roleStates);
  } else if (action.type === 'CLEAR_QUERY') {
    usermgmt.clearUsermgmtQuery();
  }
});

document.addEventListener("DOMContentLoaded", function(){
  store.dispatch(actions.initItems())
  onNavInitialize();
});

