"use strict";

// Function for gateway config page
import {formToObjectList, onNavInitialize, handleModalOnToggleById} from './common.js';
import {toggleLoadingIcon} from './common.js';
import {store, actions} from './model/gconf_model.js';

// Gateway Config Page logic
store.subscribe((action) => {
  if (action.type === 'SET_CONFIG_ITEMS') {
    const {configItems} = store.getState();
    buildGatewayConfigItems(configItems);
  }
  if (action.type === 'SET_CATEGORIES') {
    const {category, categories, exportLoading} = store.getState();
    buildGatewayConfigCategoryTabs(category, categories);
    store.dispatch(actions.getConfigItems(category))
    const exportBtn = document.getElementById("gconf-export-btn");
    exportBtn.disabled = exportLoading;
  }
  if (action.type === 'SET_EXPORT_LOADING') {
    const {exportLoading} = store.getState();
    const exportBtn = document.getElementById("gconf-export-btn");
    exportBtn.disabled = exportLoading;
    toggleLoadingIcon(exportBtn, exportLoading);
  }
});

const formItemsToChangeItems = (formItems) => {
  const {category, configItems} = store.getState();
  const changeItems = [];
  let configItem;
  formItems.forEach(itm => {
    switch (itm.action) {
      case "update":
        configItem = configItems.find(cItm => String(cItm.id) === String(itm.id));
        if (itm.value !== configItem.value) {
          changeItems.push({
            id: configItem.id,
            key: configItem.key,
            value: itm.value,
            displayName: configItem.displayName,
            category: category,
            status: configItem.status,
            validUntil: configItem.validUntil
          });
        }
        break;

      case "add":
        if (category === 0 && itm.key !== "") {
          changeItems.push({
            id: null,
            key: itm.key,
            value: itm.value,
            category: category,
            displayName: null,
            status: null,
            validUntil: null,
          });
        }
        break;

      case "delete":
        if (category === 0) {
          configItem = configItems.find(cItm => String(cItm.id) === String(itm.id));
          changeItems.push({
            id: configItem.id,
            key: configItem.key,
            value: null,
            category: category,
            displayName: configItem.displayName,
            status: configItem.status,
            validUntil: configItem.validUntil,
          });
        }
        break;
    } 
  });
  return changeItems;
};

const buildGatewayConfigCategoryTabs = (category, categories) => {
  const gconfTabs = document.getElementById("gconf-tabs");
  gconfTabs.innerHTML = "";
  const allCategories = categories.slice();
  allCategories.push({
    category: 0,
    categoryName: 'Custom',
  });

  allCategories.forEach(cat => {
    const gconfTab = document.getElementById("gconf-tab").cloneNode(true);
    const tab = gconfTab.content.querySelector("[data-gconf-item='tab']");
    tab.addEventListener('click', () => handleTabsOnClick(tab, cat.category));
    tab.textContent = cat.categoryName;
    if (cat.category === category) {
      tab.classList.add('bg-neutral-300');
    }
    gconfTabs.appendChild(gconfTab.content);
  });
};

const buildGatewayConfigItems = (results) => {
  const {category} = store.getState();
  const gconfGrid = document.getElementById("gconf-grid");
  gconfGrid.innerHTML = "";
  let validateList = [];
  const setPreviewButtonState = () => {
    const previewButton = document.getElementById("gconf-preview-btn");
    previewButton.disabled = !validateList.reduce(
      (v, elem) => v && Number(elem.getAttribute("data-gconf-validated")), true);
  }
  const validateInputOnChange = (target, pattern, pLabel) => {
    // Validate Gateway Config Item if pattern exist
    const vregexp = new RegExp(pattern, 'g');
    const validated = vregexp.test(target.value) || target.value === '';
    target.setAttribute("data-gconf-validated", validated ? 1 : 0);
    if (validated) {
      target.classList.remove('ring-2');
      target.classList.remove('outline-0');
      if (pLabel) {
        pLabel.classList.add('hidden');
      }
    } else {
      target.classList.add('ring-2');
      target.classList.add('outline-0');
      if (pLabel) {
        pLabel.classList.remove('hidden');
      }
    }
    setPreviewButtonState();
  };
  results.forEach(itm => {
    const id = itm.id;
    const gconfItem = document.getElementById("gconf-item").cloneNode(true);
    const root = gconfItem.content.querySelector("[data-gconf-item='root']");
    root.setAttribute("gconf-item-id", id);
    if (category === 0) {
      const inputLabel = gconfItem.content.querySelector("[data-gconf-item='label']");
      inputLabel.textContent = itm.key;
      const deleteButton = gconfItem.content.querySelector("[data-gconf-item='delete']");
      deleteButton.classList.remove('hidden');
      deleteButton.addEventListener('click', () => handleDeleteButtonOnClick(deleteButton));
    } else {
      const inputLabel = gconfItem.content.querySelector("[data-gconf-item='label']");
      inputLabel.textContent = itm.displayName;
    }

    // pick input with suitable format
    const inputs = gconfItem.content.querySelectorAll("[data-gconf-format]");
    Array.from(inputs).forEach(i => {
      if (itm.format && (itm.format === "boolean" || itm.format === "integer")) {
        if (i.getAttribute("data-gconf-format") !== itm.format) {
          console.log(i);
          i.remove(); 
        }
      } else {
        if (i.getAttribute("data-gconf-format") !== "any") {
          console.log(i);
          i.remove(); 
        }
      }
    });

    // Format input
    const valueInput = gconfItem.content.querySelector("[data-gconf-item='value']");
    valueInput.name = "value-" + id;
    valueInput.value = itm.value;
    if (itm.hint) {
      valueInput.placeholder = itm.hint;
    }
    if (itm.pattern !== null && itm.pattern !== undefined) {
      validateList.push(valueInput); // register
      const patternLabel = gconfItem.content.querySelector("[data-gconf-item='pattern']");
      valueInput.addEventListener('change', () => validateInputOnChange(valueInput, itm.pattern, patternLabel));
    }

    const actionInput = gconfItem.content.querySelector("[data-gconf-item='action']");
    actionInput.name = "action-" + id;
    // Add event handler
    const revertButton = gconfItem.content.querySelector("[data-gconf-item='revert']");
    revertButton.addEventListener('click', () => handleRevertButtonOnClick(revertButton));

    gconfGrid.appendChild(gconfItem.content);
  });
  if (category === 0) {
    buildOneCustomItem(gconfGrid, 0);
  }
};

const buildOneCustomItem = (gconfGrid, idx) => {
  const customItem = document.getElementById("gconf-custom").cloneNode(true);
  const root = customItem.content.querySelector("[data-gconf-item='root']");
  root.setAttribute("gconf-item-id", idx);

  const keyInput = customItem.content.querySelector("[data-gconf-item='key']");
  keyInput.name = "key-custom" + idx;
  const valueInput = customItem.content.querySelector("[data-gconf-item='value']");
  valueInput.name = "value-custom" + idx;
  const actionInput = customItem.content.querySelector("[data-gconf-item='action']");
  actionInput.name =  "action-custom" + idx;
  const addButton = customItem.content.querySelector("[data-gconf-item='add']");
  addButton.addEventListener('click', () => handleAddButtonOnClick(addButton));
  const removeButton = customItem.content.querySelector("[data-gconf-item='remove']");
  removeButton.addEventListener('click', () => handleRemoveButtonOnClick(removeButton));

  gconfGrid.appendChild(customItem.content);
};

const previewChangeItems = () => {
  const {changeItems, configItems} = store.getState();
  handleModalOnToggleById("gconf-preview-modal");
  const submitBtn = document.getElementById("gconf-post-btn");
  submitBtn.disabled = !changeItems.length;
  const tbody = document.getElementById("gconf-preview-table").getElementsByTagName("tbody")[0];
  tbody.innerHTML = "";
  changeItems.forEach(itm => {
    const rowTemplate = document.getElementById("gconf-preview-table-row-template").cloneNode(true);
    const row = rowTemplate.content.firstElementChild;
    const cols = row.children;
    const action = itm.id === null ? 'Add' : (itm.value === null ? 'Delete' : 'Update');
    cols[0].textContent = itm.displayName === null ? itm.key : itm.displayName;
    cols[1].textContent = action;
    cols[2].textContent = itm.id === null ? '' : configItems.find(cItm => cItm.id === itm.id).value;
    cols[3].textContent = '=>';
    cols[4].textContent = itm.value === null ? '' : itm.value;
    tbody.appendChild(row);
  });
};

export const handleTabsOnClick = (element, category) => {
  // Reset active tab
  const tabs = document.getElementById("gconf-tabs").querySelectorAll('[data-gconf-item="tab"]');
  tabs.forEach(tab => {
    tab.classList.remove("bg-neutral-300");
  });
  element.classList.add('bg-neutral-300');
  store.dispatch(actions.getConfigItems(category))
};

const handlePreviewButtonOnClick = () => {
  const form = document.getElementById("gconf-form");
  const formItems = formToObjectList(form);
  const changeItems = formItemsToChangeItems(formItems);
  store.dispatch(actions.setChangeItems(changeItems));
  previewChangeItems();
};

const handleSubmitButtonOnClick = (element) => {
  store.dispatch(actions.postChangeItems());
  handleModalOnToggleById("gconf-preview-modal");
};

const handleExportButtonOnClick = (element) => {
  let confirmation = confirm("Are you sure to export All Categories Gateway Configurations?");
  if (confirmation) {
    store.dispatch(actions.exportConfigItems());
  }
};

const handleDeleteButtonOnClick = (element) => {
  const root = element.closest("[data-gconf-item='root']");

  const mask = root.querySelector("[data-gconf-item='mask']");
  mask.classList.remove('opacity-100');
  mask.classList.add('opacity-40');

  const valueInput = root.querySelector("[data-gconf-item='value']");
  valueInput.setAttribute("disabled", true);

  const actionInput = root.querySelector("[data-gconf-item='action']");
  actionInput.value = "delete";

  const deleteButton = root.querySelector("[data-gconf-item='delete']");
  deleteButton.classList.add("hidden");
};

const handleRevertButtonOnClick = (element) => {
  const {category, configItems} = store.getState();

  const root = element.closest("[data-gconf-item='root']");

  const valueInput = root.querySelector("[data-gconf-item='value']");
  const actionInput = root.querySelector("[data-gconf-item='action']");

  if (actionInput.value === "update") {
    const id = root.getAttribute("gconf-item-id");
    const configItem = configItems.find(itm => String(itm.id) === id);
    valueInput.value = configItem.value;
    valueInput.dispatchEvent(new Event("change"));

  } else if (actionInput.value === "delete") {
    if (category === 0) {
      const mask = root.querySelector("[data-gconf-item='mask']");
      mask.classList.remove('opacity-40');
      mask.classList.add('opacity-100');

      const deleteButton = root.querySelector("[data-gconf-item='delete']");
      deleteButton.classList.remove("hidden");

      valueInput.removeAttribute("disabled");
      actionInput.value = "update";
    }
  }
};

const handleAddButtonOnClick = (element) => {
  const root = element.closest("[data-gconf-item='root']");
  const gconfGrid = document.getElementById("gconf-grid");
  const id = root.getAttribute("gconf-item-id");

  const addButton = root.querySelector("[data-gconf-item='add']");
  addButton.classList.add("hidden");
  const removeButton = root.querySelector("[data-gconf-item='remove']");
  removeButton.classList.remove("hidden");

  buildOneCustomItem(gconfGrid, id+1);
};

const handleRemoveButtonOnClick = (element) => {
  const root = element.closest("[data-gconf-item='root']");
  root.parentNode.removeChild(root);
};

document.addEventListener("DOMContentLoaded", function(){
  store.dispatch(actions.getCategories())
  onNavInitialize();
});
document.getElementById('gconf-preview-btn').addEventListener('click', handlePreviewButtonOnClick);
document.getElementById('gconf-post-btn').addEventListener('click', handleSubmitButtonOnClick);
document.getElementById('gconf-export-btn').addEventListener('click', handleExportButtonOnClick);

