"use strict";

import {
  checkInvalidCharacterInItem,
  checkRequiredFieldInItem,
  displayDropdownBoxLabel,
  handleDropDownOnClick,
  handleInputLabels,
  getFileManagementDateInputValue
} from './common.js';
import {getBaseUploadForm} from './component/upload_form.js';
import {getHistoryModal} from './component/history_modal.js';

const getMultilineTextAreaHandler = (inst, inputName) => (e) => {
  const rawText = e.target.value;
  const texts = [];
  rawText.split(/\r?\n/).forEach(s => {
    const t = s.trim()
    if (t) {
      texts.push(t);
    }
  });
  const labelContainer = document.querySelector(`[data-${inst.tag}-${inst.form}='${inputName}']`);
  handleInputLabels(labelContainer, rawText, e.target.tagName);
  inst.store.dispatch(inst.actions.setUploadItem({[inputName]: texts}));
};

const baseFileManagementUploadForm = (formtype, fmStore, fmActions) => {
  const inst = getBaseUploadForm(formtype, fmStore, fmActions, 'form');
  inst.store.subscribe((action) => {
    const {tab} = inst.store.getState();
    const {[inst.formTypeName]: formType} = inst.store.getState();
    if (formType !== inst.formType) {
      return;
    }
    if (action.type === 'SET_DETAIL') {
      const {detail} = inst.store.getState();
      if (detail) {
        inst.previewModal.buildModalContent(detail);
        inst.previewModal.handleHistoryModalOnToggle({});
      }
    }
  });
  inst.handleUploadItemOnChange = function(name, target) {
    // handle input label
    const labelContainer = document.querySelector(`[data-${this.tag}-${this.formType}='${name}']`);
    let value;
    if (target.tagName === 'INPUT' && target.type) {
      if (target.type === 'file') {
        value = target.files.length ? target.files[0] : null;
      } else if (target.type === 'text') {
        value = target.value;
      } else if (target.type === 'checkbox') {
        value = target.checked;
      } else if (target.type === 'datetime-local' || target.type === 'date') {
        value = getFileManagementDateInputValue(name, target.value);
      }
    } else {
      value = target.value;
    }
    handleInputLabels(labelContainer, value, target.tagName);
    // Update store value
    this.store.dispatch(this.actions.setUploadItem({[name]: value}));
  };
  inst.previewModal = Object.create(getHistoryModal());
  inst.setPreviewModal = () => {
    const modal = inst.previewModal;
    const container = document.getElementById(`${inst.tag}-detail-modal`);
    modal.setContainer(container);
    const template = document.getElementById(`preview-modal-content-template`);
    modal.setTemplate(template);
    modal.onActivate = function(params) {};
    modal.renderModalContent = function(content, detail) {
      const renderer = (container, d) => {
        // assume receiving object
        let jsonData = JSON.stringify(d, null, '    ');
        if (container) {
          container.textContent = jsonData;
        }
      };
      if (detail) {
        content.querySelector(`[data-${inst.tag}-modal='detailLoading']`)?.remove();
        const pre = content.querySelector("pre");
        if (pre) {
          renderer(pre, detail);
        }
      }
    };
    modal.renderModalFooter = function(container, detail) {}
  };
  inst.initialize = function() {
    inst.buildForm();
    inst.setPreviewModal();
  };
  return inst;
}

// Symmetric Key Enquiry
const baseSymmetricKeyEnquiryRequest = (formtype, fmStore, fmActions) => {
  const inst = baseFileManagementUploadForm(formtype, fmStore, fmActions);
  inst.renderFormItem = function(content) {
    const fileTypeSelect = content.querySelector(`[data-${inst.tag}-${inst.form}='file_type']`).querySelector("select");
    fileTypeSelect.addEventListener('change', (e) => inst.handleUploadItemOnChange('file_type', e.target));
    const filesTextarea = content.querySelector(`[data-${inst.tag}-${inst.form}='file_names']`).querySelector("textarea");
    filesTextarea.addEventListener('change', getMultilineTextAreaHandler(this, 'file_names'));
  };
  return inst;
};
export const SymmetricKeyEnquiryRequest = (fmStore, fmActions) => {
  const inst = baseSymmetricKeyEnquiryRequest('symkeyenq', fmStore, fmActions);
  let baseRenderFormItem = inst.renderFormItem;
  inst.renderFormItem = function(content) {
    baseRenderFormItem(content);
    const orgMsgIdInput = content.querySelector(`[data-${this.tag}-${this.form}='org_msg_id']`).querySelector("input");
    orgMsgIdInput.addEventListener('change', (e) => this.handleUploadItemOnChange('org_msg_id', e.target));
  };
  inst.isFormSubmittable = function() {
    const {uploadItem} = this.store.getState();
    let isValid = checkRequiredFieldInItem(uploadItem, ['file_type', 'file_names', 'org_msg_id']);
    isValid = isValid && checkInvalidCharacterInItem(uploadItem, ['org_msg_id']);
    return isValid;
  };
  return inst;
};
export const LostSymmetricKeyEnquiryRequest = (fmStore, fmActions) => {
  const inst = baseSymmetricKeyEnquiryRequest('lostkeyenq', fmStore, fmActions);
  inst.isFormSubmittable = function() {
    const {uploadItem} = this.store.getState();
    let isValid = checkRequiredFieldInItem(uploadItem, ['file_type', 'file_names']);
    return isValid;
  };
  return inst;
};
// CP / CRA Listing
export const ListingRequest = (formtype, fmStore, fmActions) => {
  const inst = baseFileManagementUploadForm(formtype, fmStore, fmActions);
  inst.renderMultiDropdown = (dropdown, field) => {
    dropdown.querySelector("[data-dropdown='box']").addEventListener('click', () => handleDropDownOnClick(dropdown));
    dropdown.querySelector("[data-dropdown='mask']").addEventListener('click', () => handleDropDownOnClick(dropdown));
    const inputs = dropdown.querySelector("[data-dropdown='container']").querySelectorAll('input');
    const target = () => {
      return {
        value: Array.from(inputs).filter(i => i.checked).map(o => o.value),
        tagName: 'SELECT',
      };
    };
    inputs.forEach(input => {
      input.addEventListener('change', () => displayDropdownBoxLabel(dropdown));
      input.addEventListener('change', () => inst.handleUploadItemOnChange(field, target()));
    });
  };
  inst.renderCheckboxGroup = (checkboxGroup, field) => {
    const inputs = checkboxGroup.querySelectorAll('input');
    const handleCheckboxGroupOnChange = (target) => {
      // toggle checkboxes
      Array.from(inputs).filter(input => input.type === 'checkbox').forEach(input => {
        if (input === target) {
          input.checked = true;
        } else {
          input.checked = false;
        }
      });
      // recreate target
      const newTarget = () => {
        return {
          value: target.value,
          tagName: 'SELECT',
        };
      };
      inst.handleUploadItemOnChange(field, newTarget())
    };
    inputs.forEach((input, idx) => {
      if (idx === 1) {
        input.checked = true;
        handleCheckboxGroupOnChange(input);
      }
      input.addEventListener('change', (e) => handleCheckboxGroupOnChange(e.target));
    });
  };
  inst.renderFormItem = function(content) {
    const ptCodeInput = content.querySelector(`[data-${this.tag}-${this.form}='pt_code']`).querySelector("textarea");
    ptCodeInput.addEventListener('change', getMultilineTextAreaHandler(this, 'pt_code'));
    const statusDropdown = content.querySelector(`[data-${this.tag}-${this.form}='status']`);
    this.renderMultiDropdown(statusDropdown, 'status');
    const groupNames = ['effect_date_time', 'full_name', 'short_name', 'maintenance'];
    groupNames.forEach(name => {
      const checkboxGroup = content.querySelector(`[data-${this.tag}-${this.form}='${name}']`);
      this.renderCheckboxGroup(checkboxGroup, name);
    });
  };
  return inst;
};

export const EncryptionCertificateEnquiryRequest = (fmStore, fmActions) => {
  const inst = baseFileManagementUploadForm('enccertenq', fmStore, fmActions);
  inst.renderCheckboxGroupWithCustom = (checkboxGroup, field) => {
    const inputs = checkboxGroup.querySelectorAll('input');
    const handleCheckboxGroupOnChange = (target) => {
      // toggle checkboxes
      let customInput = checkboxGroup.querySelector(`[data-${inst.tag}-${inst.form}='${target.value}']`)?.querySelector("input");
      Array.from(inputs).filter(input => input.type === 'checkbox').forEach(input => {
        if (input === target) {
          input.checked = true;
          if (customInput) {
            customInput.disabled = false;
          }
        } else {
          input.checked = false;
          if (!customInput) {
            Array.from(inputs).filter(input => input.type !== 'checkbox').forEach(input => {
              input.disabled = true;
            });
          }
        }
      });
      // recreate target
      const newTarget = () => {
        return {
          value: [customInput ? customInput.value : target.value],
          tagName: 'SELECT',
        };
      };
      inst.handleUploadItemOnChange(field, newTarget())
    };
    inputs.forEach((input, idx) => {
      if (idx === 0) {
        input.checked = true;
        handleCheckboxGroupOnChange(input);
      }
      if (input.type === 'checkbox') {
        input.addEventListener('change', (e) => handleCheckboxGroupOnChange(e.target));
      } else if (input.type === 'text') {
        input.addEventListener('change', (e) => {
          const value = e.target.value;
          inst.handleUploadItemOnChange('pt_code', { value: [value] });
        });
      }
    });
  };
  inst.renderFormItem = function(content) {
    const groupNames = ['pt_code'];
    groupNames.forEach(name => {
      const checkboxGroup = content.querySelector(`[data-${this.tag}-${this.form}='${name}']`);
      this.renderCheckboxGroupWithCustom(checkboxGroup, name);
    });
  };
  return inst;
};
const getSimpleBaseUploadForm = (formtype, fmStore, fmActions, inputNames) => {
  const inst = baseFileManagementUploadForm(formtype, fmStore, fmActions);
  const requiredNames = [];
  inst.renderFormItem = function(content) {
    inputNames.forEach(name => {
      const container = content.querySelector(`[data-${inst.tag}-${inst.form}='${name}']`);
      const input = container.querySelector("input");
      input.addEventListener('change', (e) => inst.handleUploadItemOnChange(name, e.target));

      // Check required
      if (!!container.querySelector("[data-form-label='required']")) {
        requiredNames.push(name);
      }
    });
  };
  inst.isFormSubmittable = function() {
    const {uploadItem} = inst.store.getState();
    let isValid = checkRequiredFieldInItem(uploadItem, requiredNames);
    return isValid;
  };
  return inst;

};
export const CertificateUploadRequest = (fmStore, fmActions) => {
  const inputNames = ['cert_type', 'cert', 'eff_start_date_time', 'eff_end_date_time']
  return getSimpleBaseUploadForm('certupload', fmStore, fmActions, inputNames);
};

export const ParticipantPublicCertificateRevocationRequest = (fmStore, fmActions) => {
  const inputNames = ['pt_code', 'cert_type', 'cert'];
  return getSimpleBaseUploadForm('ppcertrevo', fmStore, fmActions, inputNames);
};

export const ClientSecretRenewalNotification = (fmStore, fmActions) => {
  const inputNames = ['pt_code', 'client_id', 'expiry_date_time'];
  return getSimpleBaseUploadForm('csecrenew', fmStore, fmActions, inputNames);
};

export const ClientSecretAmendmentRequest = (fmStore, fmActions) => {
  const inputNames = ['pt_code', 'client_id', 'org_secret', 'new_secret'];
  return getSimpleBaseUploadForm('csecamend', fmStore, fmActions, inputNames);
};

export const MessageSigningKeyRequest = (fmStore, fmActions) => {
  const inputNames = ['pt_code'];
  return getSimpleBaseUploadForm('msgskeyreq', fmStore, fmActions, inputNames);
};

export const HistoricalMessageEnquiryRequest = (fmStore, fmActions) => {
  const inputNames = ['pt_code', 'start_date_time', 'end_date_time'];
  const inst = getSimpleBaseUploadForm('hmsgenq', fmStore, fmActions, inputNames);
  let baseRenderFormItem = inst.renderFormItem;
  inst.renderFormItem = function(content) {
    baseRenderFormItem(content);
    const msgInput = content.querySelector(`[data-${inst.tag}-${inst.form}='msg_type']`).querySelector("textarea");
    msgInput.addEventListener('change', (e) => inst.handleUploadItemOnChange('msg_type', e.target));
  };
  return inst;
};
export const HistoricalMessageResendRequest = (fmStore, fmActions) => {
  const inst = getSimpleBaseUploadForm('hmsgresend', fmStore, fmActions, []);
  let baseRenderFormItem = inst.renderFormItem;
  inst.renderFormItem = function(content) {
    baseRenderFormItem(content);
    const msgInput = content.querySelector(`[data-${inst.tag}-${inst.form}='msg_ids']`).querySelector("textarea");
    msgInput.addEventListener('change', getMultilineTextAreaHandler(this, 'msg_ids'));
  };
  inst.isFormSubmittable = function() {
    const {uploadItem} = inst.store.getState();
    let isValid = checkRequiredFieldInItem(uploadItem, ['msg_ids']);
    return isValid;
  };
  return inst;
};
export const MiscellaneousBusinessNotificationRequest = (fmStore, fmActions) => {
  const inputNames = ['messageId', 'encryptedContent', 'dataFormat', 'dataCompression', 'dataFileName'];
  const inst = getSimpleBaseUploadForm('miscbusnoti', fmStore, fmActions, inputNames);
  let baseRenderFormItem = inst.renderFormItem;
  inst.renderFormItem = function(content) {
    baseRenderFormItem(content);
    const msgInput = content.querySelector(`[data-${inst.tag}-${inst.form}='category']`).querySelector("select");
    msgInput.addEventListener('change', (e) => inst.handleUploadItemOnChange('category', e.target));
  };
  return inst;
};

export const ReportEnquiryRequest = (fmStore, fmActions) => {
  const inputNames = ['pt_code', 'rpt_id', 'rpt_date'];
  return getSimpleBaseUploadForm('rptenq', fmStore, fmActions, inputNames);
};
