"use strict";

import {store, actions} from './model/department_model.js';
import {Usermgmt} from './component/usermgmt_component.js';
import {hasInvalidCharacter, jsonObjectIsEqual, onNavInitialize} from './common.js';
import {checkRequiredFieldInItem, checkInvalidCharacterInItem} from './common.js';
import {camelCaseToSentenceCase} from './common.js';
import {getReorderedHeadersAndRecArrs} from './component/history_export.js';

const usermgmt = Usermgmt(store, actions);

usermgmt.fillUsermgmtTableRow = (cols, item, viewButton) => {
  cols[0].textContent = item.departmentCode
  cols[1].textContent = item.departmentName
  cols[2].appendChild(viewButton);
  cols[2].classList.add("text-right");
};

usermgmt.renderViewItemContent = (formTable, itm) => {
  const deptNameItem = formTable.querySelector("[data-usermgmt-detail='departmentName']");
  const deptNameInput = deptNameItem.querySelector("input");
  deptNameInput.value = itm.departmentName ? itm.departmentName : null;
  deptNameInput.addEventListener('input', (e) => usermgmt.handleFormInputOnChange('departmentName', e.target.value));

  const deptCodeItem = formTable.querySelector("[data-usermgmt-detail='departmentCode']");
  const deptCodeInput = deptCodeItem.querySelector("input");
  deptCodeInput.value = itm.departmentCode || itm.departmentCode === 0 ? itm.departmentCode : null;
  deptCodeInput.disabled = true;
};

usermgmt.renderAddItemContent = (formTable) => {
  const deptNameItem = formTable.querySelector("[data-usermgmt-detail='departmentName']");
  const deptNameInput = deptNameItem.querySelector("input");
  deptNameInput.addEventListener('input', (e) => usermgmt.handleFormInputOnChange('departmentName', e.target.value));

  const deptCodeItem = formTable.querySelector("[data-usermgmt-detail='departmentCode']");
  const deptCodeInput = deptCodeItem.querySelector("input");
  deptCodeInput.addEventListener('input', (e) => usermgmt.handleFormInputOnChange('departmentCode', e.target.value));
};

usermgmt.isAddItemButtonActive = () => {
  const {changeItem} = store.getState();
  let isValid = checkRequiredFieldInItem(changeItem, ['departmentCode', 'departmentName']);
  isValid = isValid && checkInvalidCharacterInItem(changeItem, ['departmentCode', 'departmentName']);
  return isValid;
};
usermgmt.isUpdateItemButtonActive = () => {
  const {changeItem, items} = store.getState();
  const item = items.find(itm => itm.departmentCode === changeItem.departmentCode);
  let isActive = false;
  if (item && changeItem.departmentName !== item.departmentName) {
    isActive = true;
  }
  if (!isActive) {
    alert("No updated field")
    return isActive;
  }
  isActive = isActive && checkInvalidCharacterInItem(changeItem, ['departmentName']);
  return isActive;
};
usermgmt.isSearchButtonActive = () => {
  const {queryParams} = store.getState();
  return checkInvalidCharacterInItem(queryParams, ['departmentName', 'departmentCode']);
};
usermgmt.exportRecords = () => {
  const {items, roleTree} = store.getState();
  // Deep Copy
  const records = JSON.parse(JSON.stringify(items));
  const order = ['departmentCode', 'departmentName'];
  let {headers, recArrs} = getReorderedHeadersAndRecArrs(records, order)
  headers = headers.map(camelCaseToSentenceCase);
  return {filename: 'department_export.csv', headers, recArrs};
};

store.subscribe((action) => {
  if (action.type === 'SET_INIT_ITEMS') {
    const {currentPage, itemPerPage, items, currentPageItems} = store.getState();
    usermgmt.buildUsermgmtLayout();
    usermgmt.buildUsermgmtTable(currentPageItems);
    usermgmt.buildUsermgmtPagination(currentPage, itemPerPage, items.length);
  } else if (action.type === 'SET_ITEMS' || action.type === 'SET_PAGE') {
    const {currentPage, itemPerPage, items, currentPageItems} = store.getState();
    usermgmt.buildUsermgmtTable(currentPageItems);
    usermgmt.buildUsermgmtPagination(currentPage, itemPerPage, items.length);
  } else if (action.type === 'CLEAR_QUERY') {
    usermgmt.clearUsermgmtQuery();
  }
});

document.addEventListener("DOMContentLoaded", function(){
  store.dispatch(actions.initItems())
  onNavInitialize();
});

