"use strict";

import {handleCollapseCardHeaderOnClick} from './common.js';
import {hasInvalidCharacter, isNumeric, isEmptyField} from './common.js';
import {checkRequiredFieldInItem, checkInvalidCharacterInItem} from './common.js';
import {checkHasValidFileExtension, checkNonNumericInItem} from './common.js';
import {displayDropdownBoxLabel, renderMultiDropdown} from './common.js';
import {handleInputLabels, handleModalOnToggleById, toggleLoadingIcon} from './common.js';
import {convertDatetimeToLocaleString, getFilterDateInputValue, checkStatusColor} from './common.js';
import {getBaseUploadForm} from './component/upload_form.js';
import {getHistoryTable} from './component/history_table.js';
import {getHistoryModal, renderCRAResponse, renderCRPResponse} from './component/history_modal.js';
import {getFormArrayItem} from './component/form_arrayitem.js';


export const History = (store, actions, tabname) => {
  const inst = {};
  inst.store = store;
  inst.actions = actions;
  inst.tag = store.getState().tag;
  inst.store.subscribe((action) => {
    const {tab} = inst.store.getState();
    if (tabname && tab !== tabname) {
      return;
    }
    if (action.type === 'SET_RECORDS') {
      const {records} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='table']`);
      inst.table.buildTable(container, records);
    } else if (action.type === 'SET_TOTAL_PAGES') {
      const {totalPages, page, size} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='table']`);
      inst.table.buildPagination(container, page, size, totalPages);
    } else if (action.type === 'SET_TOTAL_RECORDS') {
      const {totalRecords} = inst.store.getState();
      inst.table.setTotalRecordCount(totalRecords);
    } else if (action.type === 'SET_DETAIL') {
      const {detail} = inst.store.getState();
      inst.modal.buildModalContent(detail);
      inst.modal.buildModalFooter(detail);
      const isIdTypeMessageId = !!detail.messageId;
      if (isIdTypeMessageId) {
        inst.buildResolveModal(detail.messageId, detail.resolved);
      }
    } else if (action.type === 'SET_DETAIL_LOADING') {
      const {detailLoading} = inst.store.getState();
      const modalContainer = inst.modal.getContainer();
      toggleLoadingIcon(modalContainer, detailLoading)
    }
  });
  inst.table = Object.create(getHistoryTable());
  inst.setTable = () => {
    inst.table.fillHistoryTableRow = function(cols, rec) {
      // Handle messageId
      const messageIdElement = document.createElement('a');
      const hasMessageId = !!rec.messageId;
      messageIdElement.textContent = hasMessageId ? rec.messageId : `U${rec.id}`;
      messageIdElement.classList.add("font-medium");
      messageIdElement.classList.add("text-blue-600");
      messageIdElement.classList.add("hover:underline");
      const modalOnToggleParams = {
        idType: hasMessageId ? 'messageId' : 'uploadId',
        id: hasMessageId ? rec.messageId : rec.id,
      };
      messageIdElement.addEventListener('click', () => inst.modal.handleHistoryModalOnToggle(modalOnToggleParams));
      cols[0].appendChild(messageIdElement);
      cols[1].textContent = rec.actionType;
      cols[2].textContent = rec.originalFilename;
      cols[3].textContent = rec.fileFormat;
      cols[4].textContent = rec.dcrReferenceId;
      cols[5].textContent = rec.recordCount;
      cols[6].textContent = rec.channel;
      cols[7].textContent = rec.user;
      cols[8].textContent = rec.departmentCode;
      cols[9].textContent = convertDatetimeToLocaleString(rec.createDatetime);
      // Handle Status
      const statusSpan = document.createElement('span');
      const statusTextColor = checkStatusColor(rec.status);
      statusSpan.classList.add(`text-${statusTextColor}`);
      statusSpan.innerHTML = rec.status;
      cols[10].appendChild(statusSpan);
      let resolvedIcon;
      if (rec.resolved) {
        resolvedIcon = document.getElementById("history-table-resolved-icon").cloneNode(true).content.firstElementChild;
        cols[10].appendChild(resolvedIcon);
      }
      cols[11].textContent = rec.messageRemark;
    };
    inst.table.setPage = function(page) {
      inst.store.dispatch(inst.actions.setPage(page))
    };
    inst.table.setSize = function(size) {
      inst.store.dispatch(inst.actions.setSize(size))
    }
  };
  inst.modal = Object.create(getHistoryModal());
  inst.setModal = () => {
    const {tab} = inst.store.getState();
    const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector("#detail-modal");
    inst.modal.setContainer(container);
    const template = document.getElementById(`${tab}-modal-content-template`);
    inst.modal.setTemplate(template);
    inst.modal.onActivate = function(params) {
      inst.store.dispatch(inst.actions.getDetail(params.idType, params.id));
    };
    inst.modal.renderModalContent = function(content, detail) {
      const fields = ["messageId", "actionType", "channel", "user", "departmentCode", "approvedUserName",
                      "filename", "fileFormat", "originalFilename", "fileVersion", "fileSize",
                      "fileAction", "recordCount", "dcrReferenceId", "dcrInitiator",
                      "status", "messageRemark", "resolvedMessage", "errorMessage"];
      fields.forEach(f => {
        const input = content.querySelector(`[data-${inst.tag}-detail='${f}']`).querySelector("input");
        if (input && detail[f]) {
          input.value = detail[f];
        }
      });

      const datetimeInput = content.querySelector(`[data-${inst.tag}-detail='createDatetime']`).querySelector("input");
      datetimeInput.value = convertDatetimeToLocaleString(detail.createDatetime);

      const hasMessageId = !!detail.messageId;
      // Handle download button
      const filenameContainer = content.querySelector(`[data-${inst.tag}-detail='filename']`);
      const originalFilenameContainer = content.querySelector(`[data-${inst.tag}-detail='originalFilename']`);
      let downloadButtonContainer;
      if (hasMessageId) {
        originalFilenameContainer.querySelector(`[data-${inst.tag}-detail='download']`)?.remove();
        downloadButtonContainer = filenameContainer;
      } else {
        filenameContainer.querySelector(`[data-${inst.tag}-detail='download']`)?.remove();
        downloadButtonContainer = originalFilenameContainer;
      }

      let params = {
        idType: hasMessageId ? 'messageId' : 'uploadId',
        id: hasMessageId ? detail.messageId : detail.id,
        filename: detail.filename ? detail.filename : detail.originalFilename,
      }
      const downloadButton = downloadButtonContainer.querySelector(`[data-${inst.tag}-detail='download']`);
      downloadButton.addEventListener('click', () => inst.store.dispatch(inst.actions.getFile(params.idType, params.id, params.filename)));

      const uploadResp = content.querySelector(`[data-${inst.tag}-detail='uploadResponse']`);
      renderCRPResponse(inst.tag, uploadResp, detail.uploadResponse);

      const crpResp = content.querySelector(`[data-${inst.tag}-detail='crpResponse']`);
      renderCRPResponse(inst.tag, crpResp, detail.crpResponse);

      // Handle CRA Response
      const craRespList = content.querySelector(`[data-${inst.tag}-detail='craResponseList']`);
      craRespList.innerHTML = "";

      if (detail.craResponses && detail.craResponses.length) {
        detail.craResponses.forEach(response => {
          renderCRAResponse(inst.tag, craRespList, response);
        });
      } else {
        const noRecord = document.getElementById('history-modal-no-resp').cloneNode(true).content;
        craRespList.appendChild(noRecord);
      }
    };
    inst.modal.renderModalFooter = function(container, detail) {
      const oldResolveButton = container.querySelector(`[data-${inst.tag}-detail='resolve']`).querySelector("button");
      const resolveButton = oldResolveButton.cloneNode(true);
      if (detail.resolved || !detail.messageId) {
        resolveButton.disabled = true;
      } else {
        resolveButton.disabled = false;
        const resolveModal = document.getElementById("resolve-modal");
        resolveButton.addEventListener('click', () => handleModalOnToggleById("resolve-modal"));
      }
      oldResolveButton.replaceWith(resolveButton);
    };
  };
  inst.buildResolveModal = (messageId, resolved) => {
    const resolveModal = document.getElementById("resolve-modal");
    const resolvedMessageContainer = resolveModal.querySelector(`[data-${inst.tag}-resolve='resolvedMessage']`);
    // Set event for submit button
    const handleResolveSubmitButtonOnClick = () => {
      if (!isResolveButtonActive()) {
        return;
      }
      inst.store.dispatch(inst.actions.resolveMessage(messageId));
    };
    const isResolveButtonActive = (value = "") => {
      if (resolved) {
        alert("Message already resolved");
      }
      let isActive = resolved;
      if (value) {
        if (hasInvalidCharacter(value)) {
          alert("Invalid character in resolve message");
          return false;
        }
      }
      return true
    };
    const handleResolveMessageOnChange = (value) => {
      handleInputLabels(resolvedMessageContainer, value, "TEXTAREA");
      // const resolveButton = resolveModal.querySelector(`[data-${inst.tag}-resolve='submit']`).querySelector("button");
      // resolveButton.disabled = !isResolveButtonActive(value);
      inst.store.dispatch(inst.actions.setResolvedMessage(value));
    };
    const resolveButton = resolveModal.querySelector(`[data-${inst.tag}-resolve='submit']`).querySelector("button");
    const newResolveButton = resolveButton.cloneNode(true);
    newResolveButton.addEventListener("click", () => handleResolveSubmitButtonOnClick());
    // newResolveButton.disabled = !isResolveButtonActive();
    resolveButton.replaceWith(newResolveButton); //Remove event handler

    const resolveTextarea = resolvedMessageContainer.querySelector("input");
    const newResolveTextarea = resolveTextarea.cloneNode(true);
    newResolveTextarea.addEventListener("change", (e) => handleResolveMessageOnChange(e.target.value));
    resolveTextarea.replaceWith(newResolveTextarea);
  };
  inst.buildFilter = () => {
    const queryParams = ['messageId', 'actionType', 'fileName', 'dcrReferenceId', 'channel', 'status', 'username', 'originalFilename',
                          'createDatetimeFrom', 'createDatetimeTo', 'messageRemark', 'excludeResolved', 'departmentCode'];
    const handleQueryInputOnChange = (name, target) => {
      let value;
      if (target.tagName === 'INPUT') {
        if (target.type === 'text') {
          value = target.value;
        } else if (target.type === 'checkbox') {
          value = target.checked;
        } else if (target.type === 'datetime-local' || target.type === 'date') {
          value = getFilterDateInputValue(name, target.value);
        }
      } else if (target.tagName === 'SELECT') {
        value = target.value;
      }
      const card = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='filter']`);
      const labelContainer = card.querySelector(`[data-${inst.tag}-filter='${name}']`);
      handleInputLabels(labelContainer, value, target.tagName);
      inst.store.dispatch(inst.actions.setQuery({[name]: value}));
    };
    const handleSearchButtonOnClick = () => {
      const {recordLoading} = inst.store.getState();
      if (recordLoading) {
        return;
      }
      if (!inst.isSearchButtonActive()) {
        return;
      }
      inst.store.dispatch(inst.actions.setPage());
    };
    const handleClearButtonOnClick = () => {
      const {recordLoading} = inst.store.getState();
      if (recordLoading) {
        return;
      }
      queryParams.forEach(q => {
        const fieldContainer = card.querySelector(`[data-${inst.tag}-filter='${q}']`);
        if (!fieldContainer) {
          return;
        }
        const inputs = Array.from(fieldContainer.querySelectorAll("input"));
        if (inputs && inputs.length) {
          inputs.forEach(input => {
            if (input.type === 'checkbox') {
              input.checked = false;
            } else {
              input.value = null;
            }
          });
        }
        const select = fieldContainer.querySelector("select");
        if (select) {
          select.selectedIndex = -1;
        }
        const dropdownContainer = fieldContainer.querySelector("[data-dropdown='container']");
        if (dropdownContainer) {
          displayDropdownBoxLabel(fieldContainer)
        }
      });
      inst.store.dispatch(inst.actions.clearQuery());
      inst.store.dispatch(inst.actions.getRecords());
    };
    const buildCollapseCard = (card) => {
      const header = card.querySelector("[data-collapsecard='header']");
      header.addEventListener('click', () => handleCollapseCardHeaderOnClick(card));
    };
    const renderFilterContent = (card) => {
      queryParams.forEach(q => {
        const fieldContainer = card.querySelector(`[data-${inst.tag}-filter='${q}']`);
        const input = fieldContainer.querySelector("input");
        if (input) {
          input.addEventListener("change", (e) => handleQueryInputOnChange(q, e.target));
        }
        const select = fieldContainer.querySelector("select");
        if (select) {
          select.selectedIndex = -1;
          select.addEventListener("change", (e) => handleQueryInputOnChange(q, e.target));
        }
      });
      const multiDropdownParams = ['actionType'];
      multiDropdownParams.forEach(f => {
        const dropdown = card.querySelector(`[data-${inst.tag}-filter='${f}']`);
        if (dropdown) {
          renderMultiDropdown(dropdown, f, handleQueryInputOnChange);
        }
      });
    };
    inst.isSearchButtonActive = () => {
      const {query} = store.getState();
      // let isActive = false;
      // isActive = checkHasOneFieldInItem(query);
      const checkInvalidCharacterFields = ["messageId", "fileName", "originalFilename", "dcrReferenceId",
                                           "username", "messageRemark", "departmentCode"];
      let isActive = checkInvalidCharacterInItem(query, checkInvalidCharacterFields);
      return isActive;
    };
    const card = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='filter']`);
    buildCollapseCard(card);
    renderFilterContent(card);
    const searchButton = card.querySelector("#history-filter-search");
    searchButton.addEventListener("click", handleSearchButtonOnClick);
    // if (searchButton) {
    //   searchButton.disabled = !inst.isSearchButtonActive();
    // }
    const clearButton = card.querySelector("#history-filter-clear");
    clearButton.addEventListener("click", handleClearButtonOnClick);
  };
  inst.initialize = () => {
    const {tag} = inst.store.getState();
    inst.tag = tag;
    inst.buildFilter();
    inst.setModal();
    inst.setTable();
    store.dispatch(inst.actions.getRecords());
  };
  // Table related method
  return inst;
};

const getBaseDataSubmissionUploadForm = (dsTab, dsStore, dsActions) => {
  const inst = getBaseUploadForm(dsTab, dsStore, dsActions);
  inst.attachments = getFormArrayItem();
  inst.attachments.arrayItemOnChange = function(values) {
    inst.store.dispatch(inst.actions.setUploadItem({attachments: values}));
  };
  inst.renderFormItem = function(formContent) {
    const tabContentContainer = document.querySelector(`[data-${this.tag}='content']`);
    const attachmentContainer = formContent.querySelector(`[data-${this.tag}-${this.tab}='attachments']`).querySelector("[data-form-arrayitem='container']")
    const attachmentTemplate = tabContentContainer.querySelector("[data-form-arrayitem='item-template']")
    this.attachments.initialize(attachmentContainer, attachmentTemplate);
  };
  return Object.create(inst);
};

export const Submission = (dsStore, dsActions) => {
  const inst = getBaseDataSubmissionUploadForm('submission', dsStore, dsActions);
  inst.renderFormItem = function(content) {
    Object.getPrototypeOf(this).renderFormItem(content);
    const messageRemarkTextarea = content.querySelector(`[data-${this.tag}-${this.tab}='messageRemark']`).querySelector("input");
    messageRemarkTextarea.addEventListener('change', (e) => this.handleUploadItemOnChange('messageRemark', e.target));
    const fileFormatSelect = content.querySelector(`[data-${this.tag}-${this.tab}='fileFormat']`).querySelector("select");
    fileFormatSelect.addEventListener('change', (e) => this.handleUploadItemOnChange('fileFormat', e.target));
    fileFormatSelect.selectedIndex = -1;
    const dataFileInput = content.querySelector(`[data-${this.tag}-${this.tab}='dataFile']`).querySelector("input");
    dataFileInput.addEventListener('change', (e) => this.handleUploadItemOnChange('dataFile', e.target));
  };
  inst.isFormSubmittable = function() {
    const {uploadItem} = this.store.getState();
    let isValid = checkRequiredFieldInItem(uploadItem, ['dataFile', 'fileFormat']);
    isValid = isValid && checkInvalidCharacterInItem(uploadItem, ['messageRemark']);
    return isValid;
  };
  return inst;
};

export const Amendment = (dsStore, dsActions) => {
  const inst = getBaseDataSubmissionUploadForm('amendment', dsStore, dsActions);
  inst.renderFormItem = function(content) {
    Object.getPrototypeOf(this).renderFormItem(content);
    const messageRemarkTextarea = content.querySelector(`[data-${this.tag}-${this.tab}='messageRemark']`).querySelector("input");
    messageRemarkTextarea.addEventListener('change', (e) => this.handleUploadItemOnChange('messageRemark', e.target));
    const fileFormatSelect = content.querySelector(`[data-${this.tag}-${this.tab}='fileFormat']`).querySelector("select");
    fileFormatSelect.addEventListener('change', (e) => this.handleUploadItemOnChange('fileFormat', e.target));
    fileFormatSelect.selectedIndex = -1;
    const dataFileInput = content.querySelector(`[data-${this.tag}-${this.tab}='dataFile']`).querySelector("input");
    dataFileInput.addEventListener('change', (e) => this.handleUploadItemOnChange('dataFile', e.target));
    const dcrIdInput = content.querySelector(`[data-${this.tag}-${this.tab}='dcrReferenceId']`).querySelector("input");
    dcrIdInput.addEventListener('change', (e) => this.handleUploadItemOnChange('dcrReferenceId', e.target));
  };
  inst.isFormSubmittable = function() {
    const {uploadItem} = this.store.getState();
    let isValid = checkRequiredFieldInItem(uploadItem, ['dataFile', 'fileFormat']);
    isValid = isValid && checkInvalidCharacterInItem(uploadItem, ['messageRemark', 'dcrReferenceId']);
    return isValid;
  };
  return inst;
};

export const Initload = (dsStore, dsActions) => {
  const inst = getBaseDataSubmissionUploadForm('initload', dsStore, dsActions);
  inst.renderFormItem = function(content) {
    Object.getPrototypeOf(this).renderFormItem(content);
    const messageRemarkTextarea = content.querySelector(`[data-${this.tag}-${this.tab}='messageRemark']`).querySelector("input");
    messageRemarkTextarea.addEventListener('change', (e) => this.handleUploadItemOnChange('messageRemark', e.target));
    const fileFormatSelect = content.querySelector(`[data-${this.tag}-${this.tab}='fileFormat']`).querySelector("select");
    fileFormatSelect.addEventListener('change', (e) => this.handleUploadItemOnChange('fileFormat', e.target));
    fileFormatSelect.selectedIndex = -1;
    const dataFileInput = content.querySelector(`[data-${this.tag}-${this.tab}='dataFile']`).querySelector("input");
    dataFileInput.addEventListener('change', (e) => this.handleUploadItemOnChange('dataFile', e.target));
    const recordCountInput = content.querySelector(`[data-${this.tag}-${this.tab}='recordCount']`).querySelector("input");
    recordCountInput.addEventListener('change', (e) => this.handleUploadItemOnChange('recordCount', e.target));
  };
  inst.isFormSubmittable = function() {
    const {uploadItem} = this.store.getState();
    let isValid = checkRequiredFieldInItem(uploadItem, ['dataFile', 'fileFormat', 'recordCount']);
    isValid = isValid && checkHasValidFileExtension(uploadItem, ['dataFile'], ['zip']);
    isValid = isValid && checkInvalidCharacterInItem(uploadItem, ['messageRemark']);
    isValid = isValid && checkNonNumericInItem(uploadItem, ['recordCount']);
    return isValid;
  };
  return inst;
};
