"use strict";

import {handleCollapseCardHeaderOnClick} from '../common.js';
import {displayDropdownBoxLabel, renderMultiDropdown} from '../common.js';
import {hasInvalidCharacter, isNumeric, isEmptyField} from '../common.js';
import {checkRequiredFieldInItem, checkInvalidCharacterInItem} from '../common.js';
import {handleInputLabels, handleModalOnToggleById, toggleLoadingIcon} from '../common.js';
import {convertDatetimeToLocaleString, getFilterDateInputValue, checkStatusColor} from '../common.js';
import {getHistoryTable} from './history_table.js';

export const BaseApproval = (tabname, store, actions) => {
  const inst = {};
  inst.store = store;
  inst.actions = actions;
  inst.tag = store.getState().tag;
  inst.store.subscribe((action) => {
    const {tab} = inst.store.getState();
    if (tabname && tab !== tabname) {
      return;
    }
    if (action.type === 'SET_RECORDS') {
      const {records} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='table']`);
      inst.table.buildTable(container, records);
    } else if (action.type === 'SET_TOTAL_PAGES') {
      const {totalPages, page, size} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='table']`);
      inst.table.buildPagination(container, page, size, totalPages);
    } else if (action.type === 'SET_TOTAL_RECORDS') {
      const {totalRecords} = inst.store.getState();
      inst.table.setTotalRecordCount(totalRecords);
    }
  });
  inst.table = Object.create(getHistoryTable());
  inst.setTable = function() {
    let getModal = this.getModal;
    inst.table.fillHistoryTableRow = function(cols, rec) {
      // Handle messageId
      cols[0].textContent = rec.id;
      if (tabname === 'request') {
        const arIdElement = document.createElement('a');
        arIdElement.textContent = rec.approvalRequestId;
        arIdElement.classList.add("font-medium");
        arIdElement.classList.add("text-blue-600");
        arIdElement.classList.add("hover:underline");
        arIdElement.addEventListener('click', () => getModal(rec.requestType)?.handleHistoryModalOnToggle(rec.id));
        cols[1].appendChild(arIdElement);
      } else {
        cols[1].textContent = rec.approvalRequestId;
      }
      const reqTypeSpan = document.getElementById("approval-requestType-span").content.querySelector(`[data-approval-span='${rec.requestType}']`);
      if (reqTypeSpan) {
        cols[2].textContent = reqTypeSpan.textContent;
      } else {
        cols[2].textContent = rec.requestType;
      }
      cols[4].textContent = rec.requestUser;
      if (Array.from(cols).length <= 6) {
        cols[5].textContent = convertDatetimeToLocaleString(rec.createDatetime);
      } else {
        cols[5].textContent = rec.approveUser;
        cols[6].textContent = rec.executionStatus;
        cols[7].textContent = rec.executionErrorMsg;
        cols[8].textContent = convertDatetimeToLocaleString(rec.createDatetime);
        cols[9].textContent = convertDatetimeToLocaleString(rec.lastUpdateDatetime);
      }
      // Handle Status
      const statusSpan = document.createElement('span');
      const statusTextColor = checkStatusColor(rec.status);
      statusSpan.classList.add(`text-${statusTextColor}`);
      statusSpan.innerHTML = rec.status;
      cols[3].appendChild(statusSpan);
    };
    inst.table.setPage = function(page) {
      inst.store.dispatch(inst.actions.setPage(page))
    };
    inst.table.setSize = function(size) {
      inst.store.dispatch(inst.actions.setSize(size))
    };
  };
  inst.buildFilter = function() {
    const queryParams = ['id', 'originalId', 'approvalRequestId', 'requestType', 'requestUser', 'status',
                         'executionStatus', 'approveUser', 'createDatetimeFrom', 'createDatetimeTo'];
    const handleQueryInputOnChange = (name, target) => {
      let value;
      if (target.tagName === 'INPUT') {
        if (target.type === 'text') {
          value = target.value;
        } else if (target.type === 'checkbox') {
          value = target.checked;
        } else if (target.type === 'datetime-local' || target.type === 'date') {
          value = getFilterDateInputValue(name, target.value);
        }
      } else if (target.tagName === 'SELECT') {
        value = target.value;
      }
      const card = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='filter']`);
      const labelContainer = card.querySelector(`[data-${inst.tag}-filter='${name}']`);
      handleInputLabels(labelContainer, value, target.tagName);
      inst.store.dispatch(inst.actions.setQuery({[name]: value}));
    };
    const handleSearchButtonOnClick = () => {
      const {recordLoading} = inst.store.getState();
      if (recordLoading) {
        return;
      }
      if (!inst.isSearchButtonActive()) {
        return;
      }
      inst.store.dispatch(inst.actions.setPage());
    };
    const handleClearButtonOnClick = () => {
      const {recordLoading} = inst.store.getState();
      if (recordLoading) {
        return;
      }
      queryParams.forEach(q => {
        const fieldContainer = card.querySelector(`[data-${inst.tag}-filter='${q}']`);
        if (!fieldContainer) {
          return;
        }
        const inputs = Array.from(fieldContainer.querySelectorAll("input"));
        if (inputs && inputs.length) {
          inputs.forEach(input => {
            if (input.type === 'checkbox') {
              input.checked = false;
            } else {
              input.value = null;
            }
          });
        }
        const select = fieldContainer.querySelector("select");
        if (select) {
          select.selectedIndex = -1;
        }
        const dropdownContainer = fieldContainer.querySelector("[data-dropdown='container']");
        if (dropdownContainer) {
          displayDropdownBoxLabel(fieldContainer)
        }
      });
      inst.store.dispatch(inst.actions.clearQuery());
      inst.store.dispatch(inst.actions.getRecords());
    };
    const buildCollapseCard = (card) => {
      const header = card.querySelector("[data-collapsecard='header']");
      header.addEventListener('click', () => handleCollapseCardHeaderOnClick(card));
    };
    const renderFilterContent = (card) => {
      queryParams.forEach(q => {
        const fieldContainer = card.querySelector(`[data-${inst.tag}-filter='${q}']`);
        if (!fieldContainer) {
          return
        }
        const input = fieldContainer.querySelector("input");
        if (input) {
          input.addEventListener("change", (e) => handleQueryInputOnChange(q, e.target));
        }
        const select = fieldContainer.querySelector("select");
        if (select) {
          select.selectedIndex = -1;
          select.addEventListener("change", (e) => handleQueryInputOnChange(q, e.target));
        }
      });
      const multiDropdownParams = ['requestType', 'status', 'executionStatus'];
      multiDropdownParams.forEach(f => {
        const dropdown = card.querySelector(`[data-${inst.tag}-filter='${f}']`);
        if (dropdown) {
          renderMultiDropdown(dropdown, f, handleQueryInputOnChange);
        }
      });
    };
    inst.isSearchButtonActive = () => {
      const {query} = store.getState();
      const checkInvalidCharacterFields = ["messageId", "fileName", "dcrReferenceId", "username", "messageRemark", "departmentCode"];
      let isActive = checkInvalidCharacterInItem(query, checkInvalidCharacterFields);
      return isActive;
    };
    const card = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='filter']`);
    buildCollapseCard(card);
    renderFilterContent(card);
    const searchButton = card.querySelector("#history-filter-search");
    searchButton.addEventListener("click", handleSearchButtonOnClick);
    const clearButton = card.querySelector("#history-filter-clear");
    clearButton.addEventListener("click", handleClearButtonOnClick);
  };
  inst.getModal = function() {
    console.log("To be implemented");
    return null;
  };
  inst.initialize = function() {
    const {tag} = this.store.getState();
    this.tag = tag;
    this.buildFilter();
    this.setTable();
    store.dispatch(this.actions.getRecords());
  };
  // Table related method
  return inst;
};
