"use strict";


export const APIPrefix = '/api/v1';

export const ErrorMessage = {
  CF01: "Failed on create or update configuration",
  CF02: "Category mismatch on update configuration",
  CF03: "ID mismatch on update configuration",
  CF04: "Configuration prop key cannot be null",
  CF05: "Configuration not found for non custom category",
  CF06: "Duplicated prop key on create custom configuration",
};

// Redux store vanilla js implementation
export const createStore = (reducer, initialState) => {
  let listeners = [];
  let currentState = reducer(initialState, {});
  const debugEnabled = Array.from(document.querySelectorAll("meta")).find(elem => elem.name && elem.name === "debug")?.content

  return {
    getState: () => currentState,
    dispatch: (action) => {
      currentState = reducer(currentState, action);
      if (debugEnabled) {
        console.log(action.type, action.payload, currentState);
      }

      listeners.forEach((listener) => {
        listener(action);
      });
    },
    subscribe: (newListener) => {
      listeners.push(newListener);

      const unsubscribe = () => {
        listeners = listeners.filter((l) => l !== newListener);
      };

      return unsubscribe;
    }
  };
};

export const formToObjectList = (form) => {
  const formData = new FormData(form);
  const ret = [];
  for (const [name, value] of formData) {
    const [key, id] = name.split("-");
    const idx = ret.findIndex(itm => itm.id === id);
    let item = {};
    if (idx < 0) {
      item['id'] = id;
      item[key] = value;
      ret.push(item);
    } else {
      item = ret[idx];
      item[key] = value;
    }
  }
  return ret;
}

export const onNavInitialize = () => {
  window.addEventListener('error', (e, script) => {
    if (script.endsWith(".js")) {
      alert(`Unhandled exception: ${e.message}`);
    }
  });
  const collapsibles = document.querySelectorAll("[data-nav-item='collapsible']");
  collapsibles.forEach(c => {
    const index = c.getAttribute("data-nav-index");
    const id = 'nav-' + index;
    const wasNavCollapsed = sessionStorage.getItem(id);
    if (wasNavCollapsed === 'hidden') {
      document.getElementById(id).classList.add('hidden');
    } else {
      document.getElementById(id).classList.remove('hidden');
    }
  });

  const toggleLayout = () => {
    // Handle visible nav toggle
    const layoutAsideNav = document.getElementById("layout-aside-nav");
    layoutAsideNav.classList.toggle("w-16");
    layoutAsideNav.classList.toggle("w-80");

    // Store toggle state
    const isCondenseNav = layoutAsideNav.classList.contains("w-16");
    sessionStorage.setItem("isCondenseNav", isCondenseNav ? 'condense' : '');

    // Handle container
    const layoutAside = document.getElementById("layout-aside");
    layoutAside.classList.toggle("w-16");
    layoutAside.classList.toggle("w-80");
    const layoutContent = document.getElementById("layout-content");
    layoutContent.classList.toggle("ml-16");
    layoutContent.classList.toggle("ml-80");
    const layoutDots = document.querySelectorAll("[data-nav-item='dots']");
    layoutDots.forEach(d => {
      d.classList.toggle("hidden");
    });
    collapsibles.forEach(c => {
      const index = c.getAttribute("data-nav-index");
      const id = 'nav-' + index;
      if (layoutAsideNav.classList.contains("w-16")) {
        document.getElementById(id).classList.add('hidden');
      } else if (layoutAsideNav.classList.contains("w-80")) {
        const wasNavCollapsed = sessionStorage.getItem(id);
        if (wasNavCollapsed === 'hidden') {
          document.getElementById(id).classList.add('hidden');
        } else {
          document.getElementById(id).classList.remove('hidden');
        }
      }
    });
  };

  const wasNavCondensed = sessionStorage.getItem("isCondenseNav");
  if (wasNavCondensed) {
    toggleLayout();
  }
  // Unhide nav menu
  const unhideNav = () => {
    const layoutAsideNav = document.getElementById("layout-aside-nav");
    Array.from(layoutAsideNav.children).forEach(elem => {
      elem.classList.remove("hidden");
    });
  }
  unhideNav();


  const layoutToggleButton = document.getElementById("layout-toggle");
  layoutToggleButton.addEventListener("click", () => toggleLayout());

  const handleNavItemOnMouseenter = () => {
    const layoutAside = document.getElementById("layout-aside");
    const layoutAsideNav = document.getElementById("layout-aside-nav");
    if (layoutAside) {
      layoutAside.classList.remove("w-16");
      layoutAside.classList.add("w-80");
    }
  };
  const handleNavItemOnMouseleave = () => {
    const layoutAside = document.getElementById("layout-aside");
    const layoutAsideNav = document.getElementById("layout-aside-nav");
    if (layoutAside) {
      if (layoutAsideNav.classList.contains("w-16")) {
        layoutAside.classList.add("w-16");
      }
      if (!layoutAsideNav.classList.contains("w-80")) {
        layoutAside.classList.remove("w-80");
      }
    }
  };
  let navItems = Array.from(document.querySelectorAll("[data-nav-item='collapsible']"));
  navItems = navItems.concat(Array.from(document.querySelectorAll("[data-nav-item='item']")));
  Array.from(navItems).forEach(itm => {
    itm.addEventListener("mouseenter", () => handleNavItemOnMouseenter());
    itm.addEventListener("mouseleave", () => handleNavItemOnMouseleave());
  });

  const handleNavFullItemOnMouseenter = (itm) => {
    const layoutAsideNav = document.getElementById("layout-aside-nav");
    if (layoutAsideNav.classList.contains("w-16")) {
      const collapsible = itm.querySelector("[data-nav-item='collapsible']");
      if (collapsible) {
        const index = collapsible.getAttribute("data-nav-index");
        const id = 'nav-' + index;
        document.getElementById(id).classList.remove('hidden');
      }
    }
  };
  const handleNavFullItemOnMouseleave = (itm) => {
    const layoutAsideNav = document.getElementById("layout-aside-nav");
    if (layoutAsideNav.classList.contains("w-16")) {
      const collapsible = itm.querySelector("[data-nav-item='collapsible']");
      if (collapsible) {
        const index = collapsible.getAttribute("data-nav-index");
        const id = 'nav-' + index;
        document.getElementById(id).classList.add('hidden');
      }
    }
  };
  navItems = Array.from(document.querySelectorAll("[data-nav-item='full']"));
  Array.from(navItems).forEach(itm => {
    itm.addEventListener("mouseenter", () => handleNavFullItemOnMouseenter(itm));
    itm.addEventListener("mouseleave", () => handleNavFullItemOnMouseleave(itm));
  });

  const actuatorRestart = () => {
    let confirmation = confirm("Are you sure to restart system?");
    if (confirmation) {
      fetch(`/api/system-restart`, {
        method: 'POST',
        headers: getJsonRequestHeaderWithCsrf(),
      }).then(response => {
        if (response.redirected) {
          window.location.href = response.url;
        } else {
          return response.json();
        }
      }).then(result => {
        if (result.message) {
          alert(`${result.message}`);
        } else {
          alert('Error on restarting service');
        }
      }).catch(error => {
        console.error(error);
      });
      document.getElementById('selfservice-system').classList.add('hidden');
      document.getElementById('selfservice-user-menu').classList.add('hidden');
      document.getElementById('selfservice-mask').classList.add('hidden');
    }
  };
  let restartElem = document.getElementById("selfservice-system-restart");
  if (restartElem) {
    restartElem.addEventListener('click', actuatorRestart);
  }
  const testConn = (connName) => {
    let confirmation = confirm(`Are you sure to test ${connName} connection?`);
    if (confirmation) {
      fetch(`/api/test/${connName.toLowerCase()}-connection`, {
        method: 'POST',
        headers: getJsonRequestHeaderWithCsrf(),
      }).then(response => {
        if (response.redirected) {
          window.location.href = response.url;
        } else {
          return response.text().then(data => ({status: response.status, body: data}));
        }
      }).then(data => {
        let result;
        let msg;
        // Try parse result as json
        try {
          result = JSON.parse(data.body);
        } catch (error) {
          result = {
            error: error,
            message: data.body,
          };
        }
        // Status Code as prefix
        if (data.status === 200) {
          msg = ``;
        } else {
          msg = `[Status Code ${data.status}] `;
        }
        // Try to print message
        if (result.message) {
          msg = `${msg}${result.message}`;
        } else {
          if (data.status < 300) {
            msg = `${msg}${connName} connection test success.`;
          } else {
            if (result.error) {
              msg = `${msg}${connName} connection test error: ${result.error}.`;
            } else {
              msg = `${msg}${connName} connection test error.`;
            }
          }
        }
        alert(msg);
      }).catch(error => {
        console.error(error);
      });
      document.getElementById('selfservice-system').classList.add('hidden');
      document.getElementById('selfservice-user-menu').classList.add('hidden');
      document.getElementById('selfservice-mask').classList.add('hidden');
    }

  };
  let testOauthConnElem = document.getElementById("selfservice-test-oauth-conn");
  if (testOauthConnElem) {
    testOauthConnElem.addEventListener('click', () => testConn('OAuth'));
  }
  let testCrpConnElem = document.getElementById("selfservice-test-crp-conn");
  if (testCrpConnElem) {
    testCrpConnElem.addEventListener('click', () => testConn('CRP'));
  }
  const logout = () => {
    let confirmation = confirm("Are you sure to logout?");
    if (confirmation) {
      fetch(`/logout`, {
        method: 'POST',
        headers: getJsonRequestHeaderWithCsrf(),
      }).then(response => {
        if (response.redirected) {
          window.location.href = response.url;
        } else if (response.status === 403 || response.status === 401) {
          window.location.href = '/portal';
        }
      }).catch(error => {
        console.error(error);
      });
      document.getElementById('selfservice-system').classList.add('hidden');
      document.getElementById('selfservice-user-menu').classList.add('hidden');
      document.getElementById('selfservice-mask').classList.add('hidden');
    }

  }
  let logoutElem = document.getElementById("selfservice-system-logout");
  if (logoutElem) {
    logoutElem.addEventListener('click', logout);
  }
};

export const buildTree = () => {
  let instance = {};
  instance._root = null;
  instance._nodeMap = {};
  instance.contains = (nodeId) => {
    return (nodeId in instance._nodeMap);
  };
  instance.getNodeById = (nodeId) => {
    return instance._nodeMap[nodeId];
  };
  instance.addNew = (id, nContent) => {
    const node = buildNode(id);
    Object.keys(nContent).forEach((k) => {
      const property = nContent[k];
      node.setProperty(k, property);
    });
    instance._nodeMap[id] = node;
    return node;
  };
  instance.setRoot = (id, nContent) => {
    if (instance._root === null) {
      const root = instance.addNew(id, nContent)
      root.setParent(null);
      instance._root = root;
    } else {
      console.error('Root already exists, failed to setRoot');
    }
    return instance._root;
  };
  instance.getRoot = () => {
    return instance._root;
  };
  instance.isRoot = (node) => {
    return (node.id === instance._root.id);
  };
  instance.isLeaf = (node) => {
    return (node.getChildren().length === 0);
  };
  instance.addChildTo = (nParent, id, nContent) => {
    // Find Node from nodeMap
    let childNode;
    if (id in instance._nodeMap) {
      childNode = instance._nodeMap[id];
      Object.keys(nContent).forEach((k) => {
        const property = nContent[k];
        childNode.setProperty(k, property);
      });
    } else {
      childNode = instance.addNew(id, nContent);
    }
    nParent.addChild(childNode);
    return childNode;
  };
  instance.getDepth = (node) => {
    let depth;
    if (instance.isRoot(node)) {
      depth = 0
    } else {
      depth = 1 + instance.getDepth(node.getParent());
    }
    return depth;
  };
  instance.getAncestors = (node) => {
    const ret = [];
    let n = node;
    while (!instance.isRoot(n)) {
      let p = n.getParent()
      ret.push(p);
      n = p;
    }
    return ret;
  };
  instance.deleteLeaf = (node) => {
    if (!instance.isLeaf(node)) {
      return;
    }
    // Remove from parent
    const nParent = node.getParent();
    nParent.deleteChild(node.id);

    // Remove from nodeMap
    delete instance._nodeMap[node.id];
  };
  instance.preOrder = (node) => {
    const startNode = node ? node : instance._root;
    const ret = [];
    instance._preOrder(startNode, ret);
    return ret;
  };
  instance._preOrder = (node, ret) => {
    ret.push(node);
    const childs = node.getChildren();
    childs.forEach((child) => {
      instance._preOrder(child, ret);
    });
  };
  return instance;
};

export const buildNode = (nodeId) => {
  let instance = {};
  instance.id = nodeId;
  instance._nParent = null;
  instance._nChildren = [];
  instance._properties = {}

  instance.setParent = (parentNode) => {
    instance._nParent = parentNode;
    return instance;
  };
  instance.getParent = () => {
    return instance._nParent;
  };
  instance.addChild = (childNode) => {
    instance._nChildren.push(childNode);
    childNode.setParent(instance);
    return instance;
  };
  instance.deleteChild = (childId) => {
    instance._nChildren = instance._nChildren.filter(c => c.id !== childId);
    return;
  };
  instance.getChildren = () => {
    return instance._nChildren;
  };
  instance.setProperty = (propertyName, property) => {
    instance._properties[propertyName] = property;
    return instance;
  };
  instance.getProperty = (propertyName) => {
    // Return deep copy
    return (propertyName in instance._properties) ?
      JSON.parse(JSON.stringify(instance._properties[propertyName])) :
      null;
  };
  return instance;
};

export const shuffle = (array) => {
  let currentIndex = array.length, randomIndex;
  while (currentIndex != 0) {
    randomIndex = Math.floor(Math.random() * currentIndex);
    currentIndex--;
    [array[currentIndex], array[randomIndex]] = [array[randomIndex], array[currentIndex]];
  }
  return array;
};

export const convertUrlQueryParams = (url, params) => {
  const keys = Object.keys(params);
  if (keys.length) {
    keys.filter(k => params[k] !== undefined && params[k] !== null  && params[k] !== "").forEach((k, idx) => {
      if (idx === 0) {
        url = `${url}?`;
      } else {
        url = `${url}&`;
      }
      if (Array.isArray(params[k])) {
        params[k].forEach((v, i) => {
          url = `${url}${k}=${v}`;
          if (i !== params[k].length - 1) {
            url = `${url}&`;
          }
        });
      } else {
        url = `${url}${k}=${params[k]}`;
      }
    });
  }
  return url
};

export const arrayIsEqual = (arr1, arr2) => {
  if (!(arr1 instanceof Array && arr2 instanceof Array)) {
    return false;
  }
  if (arr1.length != arr2.length) {
    return false;
  }
  for (let i = 0; i < arr1.length; i++) {
    if (arr1[i] instanceof Array && arr2[i] instanceof Array) {
      if (!arrayIsEqual(arr1[i], arr2[i])) {
        return false;
      }
    } else if (arr1[i] !== arr2[i]) {
      return false;
    }
  }
  return true;
};

export const jsonObjectIsEqual = (obj1, obj2) => {
  if (obj1 instanceof Array && obj2 instanceof Array) {
    return arrayIsEqual(obj1, obj2);
  } else if (!(obj1 instanceof Object && obj2 instanceof Object)) {
    return (obj1 === obj2);
  }

  for (const propName in obj1) {
    if (obj1.hasOwnProperty(propName) !== obj2.hasOwnProperty(propName)) {
      return false;
    }
    if (typeof obj1[propName] !== typeof obj2[propName]) {
      return false;
    }
  }
  for (const propName in obj2) {
    if (obj1.hasOwnProperty(propName) !== obj2.hasOwnProperty(propName)) {
      return false;
    }
    if (typeof obj1[propName] !== typeof obj2[propName]) {
      return false;
    }
    if (!obj1.hasOwnProperty(propName)) {
      continue;
    }
    if (obj1[propName] instanceof Array && obj2[propName] instanceof Array) {
      if (!arrayIsEqual(obj1[propName], obj2[propName])) {
        return false;
      }
    }
    if (obj1[propName] instanceof Object && obj2[propName] instanceof Object) {
      if (!jsonObjectIsEqual(obj1[propName], obj2[propName])) {
        return false;
      }
    }
    if (obj1[propName] !== obj2[propName]) {
      return false;
    }
  }
  return true;
};

export const getChangedItem = (item, changeItem) => {
  // compare changeItem with original item
  // set a property = null if it is the same as original item
  const ret = Object.assign({}, changeItem);
  console.log(item, changeItem, ret);
  Object.keys(item).forEach(k => {
    if (jsonObjectIsEqual(item[k], changeItem[k])) {
      if ( k !== "id" ) {
        ret[k] = null;
      }
    }
  });
  console.log(item, changeItem, ret);
  return ret;
};

export const displayActionResponse = (result) => {
  let resultMsg = result && result.result ? result.result : 'Error:';
  if (result.resp_code && resultMsg === 'Error') {
    resultMsg = `${resultMsg} [${result.resp_code}]`
  }
  const errMsg = result && result.resp_msg ?
    result.resp_msg : (result && result.error ? result.error : '');
  alert(`${resultMsg} ${errMsg}`);
};

export const getRequestHeaderWithCsrf = (additionalHeader) => {
  const header = Object.assign({}, additionalHeader);

  const csrfToken = document.head.querySelector("meta[name='_csrf']")?.content;
  const csrfHeader = document.head.querySelector("meta[name='_csrf_header']")?.content;

  if (csrfHeader && csrfToken) {
    header[csrfHeader] = csrfToken;
  }

  return header;
};

export const handleDownloadResponse = (response, filename, defaultExt) => {
  if (response.ok) {
    return response.blob().then(blob => {
      downloadBlob(blob, filename, defaultExt);
    });
  } else if (response.redirected) {
    window.location.href = response.url;
  } else {
    return response.text().then(text => {
      alert(`${text}`);
    });
  }
};

export const getJsonRequestHeaderWithCsrf = (additionalHeader) => {
  const jsonHeader = {
    'Accept': 'application/json',
  };

  var header = Object.assign({}, jsonHeader, additionalHeader);

  var csrfToken = document.head.querySelector("meta[name='_csrf']")?.content;
  var csrfHeader = document.head.querySelector("meta[name='_csrf_header']")?.content;

  if (csrfHeader && csrfToken) {
    header[csrfHeader] = csrfToken;
  }

  return header;
};
export const handleModalOnToggleById = (id) => {
  const container = document.getElementById(id);
  handleModalOnToggle(container);
};
export const handleModalOnToggle = (container) =>  {
  const toggleModal = (modal) => {
    modal.classList.toggle('hidden');
    const body = document.getElementsByTagName("body")[0];
    if (modal.classList.contains('hidden')) {
      let hasModalOpened = false;
      document.querySelectorAll("[data-modal='body']").forEach(m => {
        if (!m.classList.contains('hidden')) {
          hasModalOpened = true;
        }
      });
      if (!hasModalOpened) {
        body.classList.remove('overflow-hidden');
        body.classList.remove('mr-[15px]');
      }
    } else {
      body.classList.add('overflow-hidden');
      if (window.chrome && body.scrollHeight > window.innerHeight) {
        // compensate window width shift when show/hide scrollbar
        body.classList.add('mr-[15px]');
      }
    }
  };
  toggleModal(container);
  const closeButtons = container.querySelectorAll("[data-modal='close-button']");
  closeButtons.forEach(btn => {
    const newBtn = btn.cloneNode(true);
    newBtn.addEventListener('click', () => toggleModal(container));
    btn.replaceWith(newBtn);
  });
};

export const handleDropDownOnClickById = (id) => {
  handleDropDownOnClick(document.getElementById(id));
};

export const handleDropDownOnClick = (parentElement) => {
  const container = parentElement.querySelector("[data-dropdown='container']");
  // Handle dropdown position
  if (container.classList.contains('hidden')) {
    const box = parentElement.querySelector("[data-dropdown='box']");
    const boxRect = box.getBoundingClientRect();
    const containerMaxHeight = window.innerHeight - boxRect.bottom;
    const containerWidth = boxRect.right - boxRect.left;
    container.style.transform = `translate(${boxRect.left}px, ${boxRect.bottom}px)`;
    container.style.maxHeight = containerMaxHeight < 100 ? '100px' : `${containerMaxHeight}px`;
    container.style.width = `${containerWidth}px`;
  }
  container.classList.toggle('hidden');
  const mask = parentElement.querySelector("[data-dropdown='mask']");
  mask.classList.toggle('hidden');
  const body = document.getElementsByTagName("body")[0];
  const fixed = container.closest('.fixed');

  if (container.classList.contains('hidden')) {
    if (container.parentNode.closest('.fixed') === null) {
      body.classList.remove('overflow-hidden');
      body.classList.remove('mr-[15px]');
    }
  } else {
    if (window.chrome && body.scrollHeight > window.innerHeight) {
        // compensate window width shift when show/hide scrollbar
      body.classList.add('mr-[15px]');
    }
    body.classList.add('overflow-hidden');
  }
};
export const handleViewPasswordIconOnClick = (isShow, pwInput, showIcon, hideIcon) => {
  if (!!pwInput) {
    if (isShow) {
      pwInput.type = 'text';
      showIcon.classList.add('hidden');
      hideIcon.classList.remove('hidden');
    } else {
      pwInput.type = 'password';
      showIcon.classList.remove('hidden');
      hideIcon.classList.add('hidden');
    }
  }
};

export const handleCollapseCardHeaderOnClick = (collapseCard) => {
  const chevronUp = collapseCard.querySelector("[data-collapsecard='chevron-up']");
  const chevronDown = collapseCard.querySelector("[data-collapsecard='chevron-down']");
  const content = collapseCard.querySelector("[data-collapsecard='content']");
  if (content.classList.contains("hidden")) {
    content.classList.remove("hidden");
    chevronUp.classList.remove("hidden");
    chevronDown.classList.add("hidden");
  } else {
    content.classList.add("hidden");
    chevronUp.classList.add("hidden");
    chevronDown.classList.remove("hidden");
  }
};


export const toggleLoadingIcon = (container = null, isLoading = undefined) => {
  let loadingElements
  if (container) {
    loadingElements = container.querySelectorAll("[data-loading-icon]");
  } else {
    loadingElements = document.querySelectorAll("[data-loading-icon]");
  }
  if (isLoading === undefined) {
    Array.from(loadingElements).forEach(elem => {
      elem.classList.toggle("hidden");
    });
  } else {
    Array.from(loadingElements).forEach(elem => {
      if (isLoading) {
        elem.classList.remove("hidden");
      } else {
        elem.classList.add("hidden");
      }
    });
  }
};

export const renderDisplayTableScrollButton = (displayTableContainer) => {
  // Use Resize Observer API to detech element size change
  const displayTableScrollable = displayTableContainer.querySelector("[data-table='scrollable']");
  const callback = (scrollable) => {
    const xScrollable = scrollable.scrollWidth - scrollable.clientWidth > 0;
    const oldScrollLeft = displayTableContainer.querySelector("[data-table='scroll-left']");
    const oldScrollRight = displayTableContainer.querySelector("[data-table='scroll-right']");
    oldScrollLeft.style.transform = 'none';
    oldScrollRight.style.transform = 'none';
    if (xScrollable) {
      // Handle onScroll
      const scrollLeft = oldScrollLeft.cloneNode(true);
      const scrollRight = oldScrollRight.cloneNode(true);
      oldScrollLeft.replaceWith(scrollLeft);
      oldScrollRight.replaceWith(scrollRight);
      const scrollLeftOnClick = () => {
        scrollable.scrollBy({top:0, left:-20, behavior:'smooth'});
      };
      const scrollRightOnClick = () => {
        scrollable.scrollBy({top:0, left:20, behavior:'smooth'});
      };
      scrollLeft.classList.remove('hidden');
      scrollRight.classList.remove('hidden');
      // Update scroll button style
      const thead = displayTableContainer.querySelector("thead");
      const theadRect = thead.getBoundingClientRect();
      let translateY;
      const scrollLeftRect = scrollLeft.getBoundingClientRect();
      translateY = (theadRect.bottom - theadRect.top - scrollLeftRect.bottom + scrollLeftRect.top) / 2;
      scrollLeft.style.transform = `translate(0, ${translateY}px)`;

      const scrollRightRect = scrollRight.getBoundingClientRect();
      translateY = (theadRect.bottom - theadRect.top - scrollRightRect.bottom + scrollRightRect.top) / 2;
      scrollRight.style.transform = `translate(0, ${translateY}px)`;

      scrollLeft.addEventListener('click', scrollLeftOnClick);
      scrollRight.addEventListener('click', scrollRightOnClick);
    } else {
      oldScrollLeft.classList.add('hidden');
      oldScrollRight.classList.add('hidden');
    }
  };
  const handleScrollableOnScroll = (scrollable) => {
    // console.log('Firing onScroll');
    const scrollLeft = displayTableContainer.querySelector("[data-table='scroll-left']");
    const scrollRight = displayTableContainer.querySelector("[data-table='scroll-right']");
    const scrollSize = scrollable.scrollWidth - scrollable.clientWidth;
    if (scrollable.scrollLeft === 0) {
      scrollLeft.disabled = true;
    } else {
      scrollLeft.disabled = false;
    }
    if (scrollable.scrollLeft === scrollSize) {
      scrollRight.disabled = true;
    } else {
      scrollRight.disabled = false;
    }
  }
  displayTableScrollable.addEventListener('scroll', () => handleScrollableOnScroll(displayTableScrollable));

  try {
    const scrollableObserver = new ResizeObserver(() => callback(displayTableScrollable)).observe(displayTableScrollable);
  } catch (error) {
    console.log(error);
    // Call once
    callback(displayTableScrollable);
  }
};


export const displayDropdownBoxLabelById = (element, id) => {
  displayDropdownBoxLabel(element.querySelector('#' + id));
}

export const displayDropdownBoxLabel = (parentElement) => {
  const container = parentElement.querySelector("[data-dropdown='container']");
  const boxLabel = parentElement.querySelector("[data-dropdown='box']").querySelector('p');
  const items = Array.from(container.querySelectorAll('li'));
  const itemsChecked = items.filter(item => item.querySelector("input").checked)
  boxLabel.innerHTML = "";
  // Check text length
  let boxLabelLength = 0;
  let maxItemsCheckedLength = 0;
  itemsChecked.forEach((item, idx) => {
      const span = item.querySelector("span").cloneNode(true);
      boxLabelLength += span.textContent.length;
      if (boxLabelLength <= 40) {
        maxItemsCheckedLength += 1;
      }
  });
  if (itemsChecked.length > maxItemsCheckedLength ) {
      const etcSpan = document.createElement('span');
      etcSpan.textContent = `... (${itemsChecked.length} items checked)`;
      boxLabel.appendChild(etcSpan);
  } else {
    itemsChecked.forEach((item, idx) => {
      if (idx >= maxItemsCheckedLength) {
        return;
      }
      if (idx > 0) {
        const commaSpan = document.createElement('span');
        commaSpan.textContent = ', ';
        boxLabel.appendChild(commaSpan);
      }
      const span = item.querySelector("span").cloneNode(true);
      boxLabel.appendChild(span);
    });
  }
};

export const renderMultiDropdown = (dropdown, field, queryInputOnChangeHandler) => {
  dropdown.querySelector("[data-dropdown='box']").addEventListener('click', () => handleDropDownOnClick(dropdown));
  dropdown.querySelector("[data-dropdown='mask']").addEventListener('click', () => handleDropDownOnClick(dropdown));
  const inputs = dropdown.querySelector("[data-dropdown='container']").querySelectorAll('input');
  const target = () => {
    return {
      value: Array.from(inputs).filter(i => i.checked).map(o => o.value),
      tagName: 'SELECT',
    };
  };
  inputs.forEach(input => {
    input.addEventListener('change', () => displayDropdownBoxLabel(dropdown));
    input.addEventListener('change', () => queryInputOnChangeHandler(field, target()));
  });
};

export const hasInvalidCharacter = (text) => {
  const regex = /^[a-zA-Z0-9!@#\$%\^\&*\)\(+=._-\s]+$/;
  return !regex.test(text);
};


export const hasInvalidCharacterWithChinese = (text) => {
  const regex = /^[\u4E00-\u9FFFa-zA-Z0-9!@#\$%\^\&*\)\(+=._-\s]+$/;
  return !regex.test(text);
};

export const isValidPassword = (password) => {
  const reg = /(?=.*\d)(?=.*[a-z])(?=.*[A-Z]).{8,}/;
  const consecutiveNumber = /([a-zA-Z0-9])\1{3}/;
  return (!hasInvalidCharacter(password) &&
          reg.test(password) &&
          !consecutiveNumber.test(password));
};

export const isValidEmail = (email) => {
  const reg = /^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
  return (!hasInvalidCharacter(email) && reg.test(String(email).toLowerCase()));
};

export const isAlphaNumeric = (text) => {
  const regex = /^[a-zA-Z0-9\s_-]+$/;
  return regex.test(text);
};

export const isNumeric = (text) => {
  const regex = /^[0-9]+$/;
  return regex.test(text);
};

export const isEmptyField = (value) => {
  return value === null || value === undefined || value === '';
};

export const handleInputLabels = (labelContainer, value = '', tag = 'INPUT') => {
  const input = labelContainer?.querySelector(tag.toLowerCase());
  // Always show required label
  // const requiredLabel = labelContainer?.querySelector("[data-form-label='required']");
  // if (requiredLabel) {
  //   if (input && !input.disabled && !value) {
  //     requiredLabel.classList.remove('hidden');
  //   } else {
  //     requiredLabel.classList.add('hidden');
  //   }
  // }
  const invalidCharacterLabel = labelContainer?.querySelector("[data-form-label='invalidCharacter']");
  if (invalidCharacterLabel) {
    if (input && !input.disabled && value && hasInvalidCharacter(value)) {
      invalidCharacterLabel.classList.remove('hidden');
    } else {
      invalidCharacterLabel.classList.add('hidden');
    }
  }
  const notAlphaNumericLabel = labelContainer?.querySelector("[data-form-label='notAlphaNumeric']");
  if (notAlphaNumericLabel) {
    if (input && !input.disabled && value && !isAlphaNumeric(value)) {
      notAlphaNumericLabel.classList.remove('hidden');
    } else {
      notAlphaNumericLabel.classList.add('hidden');
    }
  }
  const notNumericLabel = labelContainer?.querySelector("[data-form-label='notNumeric']");
  if (notNumericLabel) {
    if (input && !input.disabled && value && !isNumeric(value)) {
      notNumericLabel.classList.remove('hidden');
    } else {
      notNumericLabel.classList.add('hidden');
    }
  }
  const invalidPasswordLabel = labelContainer?.querySelector("[data-form-label='invalidPassword']");
  if (invalidPasswordLabel) {
    if (input && !input.disabled && value && !isValidPassword(value)) {
      invalidPasswordLabel.classList.remove('hidden');
    } else {
      invalidPasswordLabel.classList.add('hidden');
    }
  }
  const invalidEmailLabel = labelContainer?.querySelector("[data-form-label='invalidEmail']");
  if (invalidEmailLabel) {
    if (input && !input.disabled && value && !isValidEmail(value)) {
      invalidEmailLabel.classList.remove('hidden');
    } else {
      invalidEmailLabel.classList.add('hidden');
    }
  }
};
export const checkRequiredFieldInItem = (item, checkFields) => {
  let isValid = true;
  const missingFields = [];
  checkFields.forEach(f => {
    if (Array.isArray(item[f])) {
      isValid = isValid && item[f] && item[f].length > 0
    } else {
      isValid = isValid && item[f]
    }
    if (!isValid) {
      missingFields.push(f);
    }
  });
  if (missingFields.length) {
    alert(`Missing required field ${missingFields.join(', ')}`);
  }
  return isValid;
};
export const checkHasOneFieldInItem = (item) => {
  let isValid = false;
  Object.keys(item).forEach(k => {
    if (item[k]) {
      isValid = true;
    }
  });
  if (!isValid) {
    alert(`Please provide at least one field`);
  }
  return isValid;
};
export const checkInvalidCharacterInItem = (item, checkFields) => {
  let isValid = true;
  const invalidFields = [];
  checkFields.forEach(f => {
    if (item[f] && hasInvalidCharacter(item[f])) {
      isValid = false;
      invalidFields.push(f);
    }
  });
  if (invalidFields.length) {
    alert(`Invalid character for field ${invalidFields.join(', ')}`);
  }
  return isValid;
};
export const checkNonNumericInItem = (item, checkFields) => {
  let isValid = true;
  const invalidFields = [];
  checkFields.forEach(f => {
    if (item[f] && !isNumeric(item[f])) {
      isValid = false;
      invalidFields.push(f);
    }
  });
  if (invalidFields.length) {
    alert(`Non numeric character in field ${invalidFields.join(', ')}`);
  }
  return isValid;
};
export const checkInvalidPasswordInItem = (item, checkFields) => {
  let isValid = true;
  const invalidFields = [];
  checkFields.forEach(f => {
    if (item[f] && !isValidPassword(item[f])) {
      isValid = false;
      invalidFields.push(f);
    }
  });
  if (invalidFields.length) {
    alert(`Invalid password format for field ${invalidFields.join(', ')}`);
  }
  return isValid;
};
export const checkInvalidEmailInItem = (item, checkFields) => {
  let isValid = true;
  const invalidFields = [];
  checkFields.forEach(f => {
    if (item[f] && !isValidEmail(item[f])) {
      isValid = false;
      invalidFields.push(f);
    }
  });
  if (invalidFields.length) {
    alert(`Invalid email format for field ${invalidFields.join(', ')}`);
  }
  return isValid;
};
export const checkHasValidFileExtension = (item, checkFields, typelist) => {
  let isValid = true;
  const invalidFields = [];
  const hasValidFileType = (file) => {
    const pattern = /\.[0-9a-z]+$/;
    const fileExtMatch = file.name.match(pattern);
    const fileExt = fileExtMatch ? fileExtMatch[0] : '';
    const type = file && file.type ?
        getFileExtensionFromMimeType(file.type) : '';
    return (type && typelist.includes(type)) ||
        (fileExt && typelist.includes(fileExt));
  };

  checkFields.forEach(f => {
    if (item[f] && !hasValidFileType(item[f])) {
      isValid = false;
      invalidFields.push(f);
    }
  });
  if (invalidFields.length) {
    alert(`File type must be one of [${typelist.join(', ')}] for field ${invalidFields.join(', ')}`);
  }
  return isValid;
};

export const convertDatetimeToLocaleDate = (datetime) => {
  if (datetime) {
    const d = new Date(datetime);
    return d.toLocaleDateString();
  } else {
    return null;
  }
};
export const convertDatetimeToLocaleString = (datetime) => {
  if (datetime) {
    const d = new Date(datetime);
    return d.toLocaleString();
  } else {
    return null;
  }
};
export const DateDiff = {

    inDays: function(d1, d2) {
        var t2 = d2.getTime();
        var t1 = d1.getTime();

        return Math.floor((t2-t1)/(24*3600*1000));
    },

    inWeeks: function(d1, d2) {
        var t2 = d2.getTime();
        var t1 = d1.getTime();

        return parseInt((t2-t1)/(24*3600*1000*7));
    },

    inMonths: function(d1, d2) {
        var d1Y = d1.getFullYear();
        var d2Y = d2.getFullYear();
        var d1M = d1.getMonth();
        var d2M = d2.getMonth();

        return (d2M+12*d2Y)-(d1M+12*d1Y);
    },

    inYears: function(d1, d2) {
        return d2.getFullYear()-d1.getFullYear();
    }
};

export const checkStatusColor = (status) => {
  const greenStatus = ['QUEUED', 'SUBMITTED', 'COMPLETED', 'SUCCESS'];
  const greyStatus = ['CREATED', 'PENDING_APPROVAL', 'PENDING'];
  const redStatus = ['ACTION_REQUIRED', 'ERROR', 'APPROVAL_REJECTED', 'EXPIRED', 'FAILED'];
  if (greenStatus.indexOf(status) > -1) {
    return 'green-600';
  } else if (greyStatus.indexOf(status) > -1) {
    return 'green-400';
  } else if (redStatus.indexOf(status) > -1) {
    return 'red-600';
  } else {
    return '';
  }
};

export const downloadBlob = (blob, name = 'file.txt', defaultExt = '') => {
  // https://dev.to/nombrekeff/download-file-from-blob-21ho
  // Convert your blob into a Blob URL (a special url that points to an object in the browser's memory)
  if (blob) {
    const pattern = /\.[0-9a-z]+$/;
    const fileExtMatch = name.match(pattern);
    const fileExt = fileExtMatch ? fileExtMatch[0] : defaultExt;
    const mime = fileExt ? getMimeTypeFromFileExtension(fileExt) : 'application/octet-stream';
    blob = blob.slice(0, blob.size, mime);
  }
  const blobUrl = URL.createObjectURL(blob);

  // Create a link element
  const link = document.createElement("a");

  // Set link's href to point to the Blob URL
  link.href = blobUrl;
  link.download = name;

  // Append link to the body
  document.body.appendChild(link);

  // Dispatch click event on the link
  // This is necessary as link.click() does not work on the latest firefox
  link.dispatchEvent(
    new MouseEvent('click', {
      bubbles: true,
      cancelable: true,
      view: window
    })
  );

  // Remove link from body
  document.body.removeChild(link);
};

export const b64ToBlob = (b64Data, contentType='', sliceSize=512) => {
  // Copy from https://stackoverflow.com/questions/16245767/creating-a-blob-from-a-base64-string-in-javascript
  const byteCharacters = atob(b64Data);
  const byteArrays = [];

  for (let offset = 0; offset < byteCharacters.length; offset += sliceSize) {
    const slice = byteCharacters.slice(offset, offset + sliceSize);

    const byteNumbers = new Array(slice.length);
    for (let i = 0; i < slice.length; i++) {
      byteNumbers[i] = slice.charCodeAt(i);
    }

    const byteArray = new Uint8Array(byteNumbers);
    byteArrays.push(byteArray);
  }

  const blob = new Blob(byteArrays, {type: contentType});
  return blob;
};

export const getFileExtensionFromMimeType = (mimeType) => {
  const types  = {
    "text/html" :"html",
    "text/css" :"css",
    "text/xml" :"xml",
    "image/gif" :"gif",
    "image/jpeg" :"jpg",
    "application/x-javascript" :"js",
    "application/atom+xml" :"atom",
    "application/rss+xml" :"rss",
    "text/mathml" :"mml",
    "text/plain" :"txt",
    "text/vnd.sun.j2me.app-descriptor" :"jad",
    "text/vnd.wap.wml" :"wml",
    "text/x-component" :"htc",
    "image/png" :"png",
    "image/tiff" :"tiff",
    "image/vnd.wap.wbmp" :"wbmp",
    "image/x-icon" :"ico",
    "image/x-jng" :"jng",
    "image/x-ms-bmp" :"bmp",
    "image/svg+xml" :"svg",
    "image/webp" :"webp",
    "application/java-archive" :"jar",
    "application/mac-binhex40" :"hqx",
    "application/msword" :"doc",
    "application/pdf" :"pdf",
    "application/postscript" :"ps",
    "application/rtf" :"rtf",
    "application/vnd.ms-excel" :"xls",
    "application/vnd.ms-powerpoint" :"ppt",
    "application/vnd.wap.wmlc" :"wmlc",
    "application/vnd.google-earth.kml+xml" :"kml",
    "application/vnd.google-earth.kmz" :"kmz",
    "application/x-7z-compressed" :"7z",
    "application/x-cocoa" :"cco",
    "application/x-java-archive-diff" :"jardiff",
    "application/x-java-jnlp-file" :"jnlp",
    "application/x-makeself" :"run",
    "application/x-perl" :"pl",
    "application/x-pilot" :"pdb",
    "application/x-rar-compressed" :"rar",
    "application/x-redhat-package-manager" :"rpm",
    "application/x-sea" :"sea",
    "application/x-shockwave-flash" :"swf",
    "application/x-stuffit" :"sit",
    "application/x-tcl" :"tcl",
    "application/x-x509-ca-cert" :"pem",
    "application/x-xpinstall" :"xpi",
    "application/xhtml+xml" :"xhtml",
    "application/x-zip-compressed" :"zip",
    "application/zip" :"zip",
    "audio/midi" :"mid",
    "audio/mpeg" :"mp3",
    "audio/ogg" :"ogg",
    "audio/x-realaudio" :"ra",
    "video/3gpp" :"3gp",
    "video/mpeg" :"mpg",
    "video/quicktime" :"mov",
    "video/x-flv" :"flv",
    "video/x-mng" :"mng",
    "video/x-ms-asf" :"asf",
    "video/x-ms-wmv" :"wmv",
    "video/x-msvideo" :"avi",
    "video/mp4" :"mp4",
  }
  return types[mimeType] ? types[mimeType] : '';
};

export const getMimeTypeFromFileExtension = (fileExtension) => {
  const types  = {
    "html": "text/html",
    "css": "text/css",
    "xml": "text/xml",
    "gif": "image/gif",
    "jpg": "image/jpeg",
    "js": "application/x-javascript",
    "atom": "application/atom+xml",
    "rss": "application/rss+xml",
    "mml": "text/mathml",
    "txt": "text/plain",
    "jad": "text/vnd.sun.j2me.app-descriptor",
    "wml": "text/vnd.wap.wml",
    "htc": "text/x-component",
    "png": "image/png",
    "tiff": "image/tiff",
    "wbmp": "image/vnd.wap.wbmp",
    "ico": "image/x-icon",
    "jng": "image/x-jng",
    "bmp": "image/x-ms-bmp",
    "svg": "image/svg+xml",
    "webp": "image/webp",
    "jar": "application/java-archive",
    "hqx": "application/mac-binhex40",
    "doc": "application/msword",
    "pdf": "application/pdf",
    "ps": "application/postscript",
    "rtf": "application/rtf",
    "xls": "application/vnd.ms-excel",
    "ppt": "application/vnd.ms-powerpoint",
    "wmlc": "application/vnd.wap.wmlc",
    "kml": "application/vnd.google-earth.kml+xml",
    "kmz": "application/vnd.google-earth.kmz",
    "7z": "application/x-7z-compressed",
    "cco": "application/x-cocoa",
    "jardiff": "application/x-java-archive-diff",
    "jnlp": "application/x-java-jnlp-file",
    "run": "application/x-makeself",
    "pl": "application/x-perl",
    "pdb": "application/x-pilot",
    "rar": "application/x-rar-compressed",
    "rpm": "application/x-redhat-package-manager",
    "sea": "application/x-sea",
    "swf": "application/x-shockwave-flash",
    "sit": "application/x-stuffit",
    "tcl": "application/x-tcl",
    "pem": "application/x-x509-ca-cert",
    "xpi": "application/x-xpinstall",
    "xhtml": "application/xhtml+xml",
    "zip": "application/zip",
    "mid": "audio/midi",
    "mp3": "audio/mpeg",
    "ogg": "audio/ogg",
    "ra": "audio/x-realaudio",
    "3gp": "video/3gpp",
    "mpg": "video/mpeg",
    "mov": "video/quicktime",
    "flv": "video/x-flv",
    "mng": "video/x-mng",
    "asf": "video/x-ms-asf",
    "wmv": "video/x-ms-wmv",
    "avi": "video/x-msvideo",
    "mp4": "video/mp4",
  }
  return types[fileExtension] ? types[fileExtension] : '';
};

export const camelCaseToSentenceCase = (text) => {
  console.log(text);
  const result = text.replace(/([A-Z])/g, " $1");
  return result.charAt(0).toUpperCase() + result.slice(1);
};

export const getFilterDateInputValue = (name, value, isLocal = false) => {
  const isDate = (value) => {
    const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
    return dateRegex.test(value);
  };
  let ret = value;
  if (value) {
    let offset = new Date().getTimezoneOffset();
    ret = new Date(value);
    if (isDate(value)) {
      const regex = /^.*To$/g;
      if (name.match(regex)) {
        ret.setUTCHours(23);
        ret.setMinutes(isLocal ? 59 : offset+59);
        ret.setSeconds(59);
      } else {
        if (!isLocal) {
          ret.setMinutes(offset);
        }
      }
    }
    if (isLocal) {
      console.log(offset);
      const YYYY = String(ret.getUTCFullYear()).padStart(4, '0');
      const MM = String(ret.getUTCMonth()+1).padStart(2, '0');
      const DD = String(ret.getUTCDate()).padStart(2, '0');
      const HH = String(ret.getUTCHours()).padStart(2, '0');
      const mm = String(ret.getUTCMinutes()).padStart(2, '0');
      const ss = String(ret.getUTCSeconds()).padStart(2, '0');
      const sss = String(ret.getUTCMilliseconds()).padStart(3, '0');
      const sign = offset <= 0 ? '%2B' : '%2D';
      const tzmm = String(Math.abs(offset) % 60).padStart(2, '0');
      const tzhh = String(Math.floor(Math.abs(offset) / 60)).padStart(2, '0');
      ret = `${YYYY}-${MM}-${DD}T${HH}:${mm}:${ss}.${sss}${sign}${tzhh}:${tzmm}`;
    } else {
      ret = ret.toISOString();
    }
  }
  return ret;
};

export const getFileManagementDateInputValue = (name, value, returnDate = false) => {
  // return browser input displayed local time as HKT
  const isDate = (value) => {
    const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
    return dateRegex.test(value);
  };
  let ret = value;
  if (value) {
    ret = new Date(value);

    const YYYY = String(ret.getFullYear()).padStart(4, '0');
    const MM = String(ret.getMonth()+1).padStart(2, '0');
    const DD = String(ret.getDate()).padStart(2, '0');
    const HH = String(ret.getHours()).padStart(2, '0');
    const mm = String(ret.getMinutes()).padStart(2, '0');
    const ss = String(ret.getSeconds()).padStart(2, '0');
    if (!isDate(value)) {
      ret = `${YYYY}-${MM}-${DD}T${HH}:${mm}:${ss}`;
    } else {
      ret = `${YYYY}-${MM}-${DD}`
    }
  }
  return ret;

}
