"use strict";

import {convertDatetimeToLocaleString, displayDropdownBoxLabelById, handleDropDownOnClickById} from './common.js';
import {getFileExtensionFromMimeType} from './common.js';
import {b64ToBlob, downloadBlob} from './common.js';
import {buildRoleCheckboxes, HandleCheckboxOnClick, renderStates} from './component/roletree.js';
import {getDetailTypeFromMessageType} from './model/business_model.js';

export const getDXULRenderer = (inst, reqType) => {
  const ret =  function(content, detail) {
    const fields = ["actionType", "channel", "user", "departmentCode",
                    "recordCount", "dcrReferenceId", "dcrInitiator",
                    "originalFilename", "status", "messageRemark"];
    fields.forEach(f => {
      const inputContainer = content.querySelector(`[data-${inst.tag}-detail='${f}']`);
      if (reqType === 'DSUL' && inputContainer) {
        if (f === 'dcrReferenceId' || f === 'dcrInitiator') {
          inputContainer.remove();
        }
      } 
      if (inputContainer) {
        const input = inputContainer.querySelector("input");
        input.value = detail[f];
      }
    });

    let params = {
      requestType: reqType,
      id: detail.id,
      filename: detail.originalFilename,
    };
    const downloadButton = content.querySelector(`[data-${inst.tag}-detail='download']`);
    downloadButton.addEventListener('click', () => inst.store.dispatch(inst.actions.getFile(params)));

    const datetimeInput = content.querySelector(`[data-${inst.tag}-detail='createDatetime']`).querySelector("input");
    datetimeInput.value = convertDatetimeToLocaleString(detail.createDatetime);
  };
  return ret;
};

export const getCRULRenderer = (inst, reqType) => {
  const ret =  function(content, detail) {
    const fields = ["actionType", "channel", "user", "departmentCode", "originalFilename",
                    "recordCount", "targetedCRAs", "status", "messageRemark"];
    fields.forEach(f => {
      const inputContainer = content.querySelector(`[data-${inst.tag}-detail='${f}']`)
      if (inputContainer) {
        const input = inputContainer.querySelector("input");
        input.value = detail[f];
      }
    });

    let params = {
      requestType: reqType,
      id: detail.id,
      filename: detail.originalFilename,
    };
    const downloadButton = content.querySelector(`[data-${inst.tag}-detail='download']`);
    downloadButton.addEventListener('click', () => inst.store.dispatch(inst.actions.getFile(params)));

    const datetimeInput = content.querySelector(`[data-${inst.tag}-detail='createDatetime']`).querySelector("input");
    datetimeInput.value = convertDatetimeToLocaleString(detail.createDatetime);
  };
  return ret;
};

export const getDCRRRenderer = (inst) => {
  const ret = function(content, detail) {
    const msgIdInput = content.querySelector(`[data-${inst.tag}-reply='messageId']`)?.querySelector("input");
    if (msgIdInput && detail.messageId) {
      msgIdInput.value = detail.messageId
    }
    const resultCodeSelect = content.querySelector(`[data-${inst.tag}-reply='resultCode']`)?.querySelector("select");
    if (resultCodeSelect && detail.resultCode) {
      resultCodeSelect.value = detail.resultCode;
    }
    const replyTypeSelect = content.querySelector(`[data-${inst.tag}-reply='replyType']`)?.querySelector("select");
    if (replyTypeSelect && detail.replyFileMimeType) {
      replyTypeSelect.value = detail.replyText ? 'replyData' : 'replyDataFile';
    }
    // Handle reply data
    const replyFileNameContainer = content.querySelector(`[data-${inst.tag}-reply='replyFileName'`);
    const replyTextContainer = content.querySelector(`[data-${inst.tag}-reply='replyText'`);
    const downloadButton = content.querySelector(`[data-${inst.tag}-reply='download'`);
    if (detail.replyText) {
      replyFileNameContainer?.remove();
      downloadButton?.remove();
      const replyTextTextarea = replyTextContainer?.querySelector("textarea");
      if (replyTextTextarea && detail.replyText) {
        replyTextTextarea.value = detail.replyText;
      }
    } else {
      replyTextContainer?.remove();
      const replyFileNameInput = replyFileNameContainer?.querySelector("input");
      if (replyFileNameInput && detail.replyFileName) {
        replyFileNameInput.value = detail.replyFileName;
      }
      if (downloadButton && detail.replyEncodedFileBytes) {
        const fileDownloader = () => {
          const b64Message = detail.replyEncodedFileBytes;
          const blob = b64ToBlob(b64Message, detail.replyFileMimeType);
          downloadBlob(blob, detail.replyFileName);
        };
        downloadButton.addEventListener('click', () => fileDownloader());
      }
    }
  };
  return ret;
};

export const getSUGXRenderer = (inst) => {
  const handleCheckboxOnClick = HandleCheckboxOnClick(inst.store, inst.actions);
  const ret = function(content, itm) {
    const formTable = content.querySelector("table");
    const groupNameItem = formTable.querySelector(`[data-${inst.tag}-detail='groupName']`);
    const groupNameInput = groupNameItem.querySelector("input");
    groupNameInput.value = itm.groupName ? itm.groupName : null;

    const groupIdItem = formTable.querySelector(`[data-${inst.tag}-detail='id']`);
    const groupIdInput = groupIdItem.querySelector("input");
    groupIdInput.name = 'id';
    groupIdInput.value = itm.groupId ? itm.groupId : null;
    groupIdInput.disabled = true;
    
    const {roleTree, detail} = inst.store.getState();
    buildRoleCheckboxes(roleTree, handleCheckboxOnClick, formTable);
    if (detail.roleStates) {
      renderStates(detail.roleStates, formTable);
    }
  };
  return ret;
};

export const getSDPXRenderer = (inst) => {
  const ret = function(content, itm) {
    const formTable = content.querySelector("table");
    const deptNameItem = formTable.querySelector(`[data-${inst.tag}-detail='departmentName']`);
    const deptNameInput = deptNameItem.querySelector("input");
    deptNameInput.value = itm.departmentName ? itm.departmentName : null;

    const deptCodeItem = formTable.querySelector(`[data-${inst.tag}-detail='departmentCode']`);
    const deptCodeInput = deptCodeItem.querySelector("input");
    deptCodeInput.value = itm.departmentCode || itm.departmentCode === 0 ? itm.departmentCode : null;
    deptCodeInput.disabled = true;
  };
  return ret;
};

export const getSUSXRenderer = (inst, reqType) => {
  const ret = function(content, itm) {
    const formTable = content.querySelector("table");
    // Handle form
    const {departments, userGroups} = itm;

    // username
    const usernameInput = formTable.querySelector(`[data-${inst.tag}-detail='username']`).querySelector("input");
    usernameInput.value = itm.username ? itm.username : null;

    // change password
    if (!itm.changePassword || reqType === 'SUSS') {
      const pwContainer = formTable.querySelector(`[data-${inst.tag}-detail='changePassword']`);
      if (pwContainer) {
        pwContainer.remove();
      }
    } else {
      const pwContainer = formTable.querySelector(`[data-${inst.tag}-detail='password']`);
      const pwInput = pwContainer.querySelector("input");
      pwInput.disabled = true;
      pwInput.type = 'password';
      pwInput.value = "********";
      const isNewPwCell = formTable.querySelector(`[data-${inst.tag}-detail='isNewPassword']`);
      Array.from(isNewPwCell.children).forEach(child => {
        if (!child.hasAttribute("data-usermgmt-detail-isnewpw")) {
          child.remove();
        }
      });
      const isNewPwCheckbox = isNewPwCell.querySelector("input");
      isNewPwCheckbox.checked = true;
    }

    // id
    const idContainer = formTable.querySelector(`[data-${inst.tag}-detail='id']`);
    if (idContainer && itm.id) {
      const idInput = idContainer.querySelector("input");
      idInput.value = itm.id ? itm.id : null;
      idInput.disabled = true;
    } else {
      idContainer.remove();
    }

    // status
    const statusParts = formTable.querySelector(`[data-${inst.tag}-detail='status']`);
    if (statusParts && !itm.status) {
      statusParts.remove();
    } else {
      const statusActive = statusParts.querySelector(`[data-${inst.tag}-status='active']`);
      const statusActiveBtn = statusActive.querySelector("Button");
      statusActiveBtn.remove();
      const statusDisabled = statusParts.querySelector(`[data-${inst.tag}-status='disabled']`);
      const statusDisabledBtn = statusDisabled.querySelector("Button");
      statusDisabledBtn.remove();
      const statusLocked = statusParts.querySelector(`[data-${inst.tag}-status='locked']`);
      const statusLockedBtn = statusLocked.querySelector("Button");
      statusLockedBtn.remove();
      if (reqType === 'SUSS' && (itm.originalStatus !== null && itm.originalStatus !== undefined)) {
        const statusElementContainer = statusActive.parentNode;
        const statusElements = {
          1: statusActive,
          0: statusDisabled,
          9: statusLocked,
        };
        const statusArrow = document.createElement("span");
        statusArrow.textContent = "=>"
        const originalStatusElem = statusElements[itm.originalStatus].cloneNode(true);
        const newStatusElem = statusElements[itm.status].cloneNode(true);
        // Clear statusElementContainer
        statusActive.remove();
        statusDisabled.remove();
        statusLocked.remove();
        // originalStatus
        statusElementContainer.appendChild(originalStatusElem);
        statusElementContainer.appendChild(statusArrow);
        statusElementContainer.appendChild(newStatusElem);
      } else {
        if (itm.status === 0) {
          statusActive.classList.add('hidden');
          statusDisabled.classList.remove('hidden');
          statusLocked.classList.add('hidden');
        } else if (itm.status === 1) {
          statusActive.classList.remove('hidden');
          statusDisabled.classList.add('hidden');
          statusLocked.classList.add('hidden');
        } else if (itm.status === 9) {
          statusActive.classList.add('hidden');
          statusDisabled.classList.add('hidden');
          statusLocked.classList.remove('hidden');
        } else {
          console.error("Unknown itm status");
        }
      }
    }

    // email 
    const emailContainer = formTable.querySelector(`[data-${inst.tag}-detail='email']`);
    if (emailContainer && reqType === 'SUSS') {
      emailContainer.remove();
    } else {
      const emailInput = emailContainer.querySelector("input");
      emailInput.value = itm.email ? itm.email : null;
    }

    // departments
    const deptContainer = formTable.querySelector(`[data-${inst.tag}-detail='departmentCode']`);
    if (deptContainer && reqType === 'SUSS') {
      deptContainer.remove();
    } else {
      const deptSelect = deptContainer.querySelector("select")
      departments.forEach(dept => {
        const child = document.createElement('option');
        child.value = dept.departmentCode;
        child.innerHTML = `${dept.departmentName}  [${dept.departmentCode}]`;
        if (dept.departmentCode === itm.departmentCode) {
          child.selected = true;
        }
        deptSelect.appendChild(child);
      });
    }

    if (reqType === 'SUSS') {
      const groupContainer = formTable.querySelector(`[data-${inst.tag}-detail='groupIds']`);
      if (groupContainer) {
        groupContainer.remove();
      }
    } else {
      const groupSelect = formTable.querySelector(`#${inst.tag}-user-groups`);
      const groupBox = groupSelect.querySelector("[data-dropdown='box']")
      groupSelect.querySelector("[data-dropdown='box']").addEventListener('click', () => handleDropDownOnClickById(`${inst.tag}-user-groups`));
      groupSelect.querySelector("[data-dropdown='mask']").addEventListener('click', () => handleDropDownOnClickById(`${inst.tag}-user-groups`));
      const groupList = groupSelect.querySelector("[data-dropdown='container']").querySelector("ul");
      const dropdownItemTemplate = formTable.querySelector(`[data-${inst.tag}-detail='groupIds']`).querySelector("template");
      userGroups.forEach((g, idx) => {
        const option = dropdownItemTemplate.cloneNode(true).content;
        const input = option.querySelector("input");
        input.value = g.id;
        option.querySelector("span").textContent = g.groupName;
        if (itm.groupIds.findIndex(id => id === g.id) >= 0) {
          input.checked = true;
        }
        groupList.appendChild(option);
      });
      displayDropdownBoxLabelById(formTable, `${inst.tag}-user-groups`);
    }

  };
  return ret;
};

export const getPXCRRenderer = (inst, reqType) => {
  const ret =  function(content, detail) {
    const fields = ["messageId", "source", "fileFormat", "fileSize", "status"];
    fields.forEach(f => {
      const inputContainer = content.querySelector(`[data-${inst.tag}-detail='${f}']`);
      if (inputContainer) {
        const input = inputContainer.querySelector("input");
        input.value = detail[f];
      }
    });

    const msgtypeSpan = document.getElementById(`business-history-span-template`).cloneNode(true).content;
    const businessDetailType = getDetailTypeFromMessageType(detail.messageType);
    const messageTypeInput = content.querySelector(`[data-${inst.tag}-detail='messageType']`).querySelector("input");
    messageTypeInput.value = Array.from(msgtypeSpan.children).find(c => c.getAttribute(`data-business-span`) === businessDetailType).textContent;

    const lastUpdateDatetimeInput = content.querySelector(`[data-${inst.tag}-detail='lastUpdateDatetime']`)?.querySelector("input");
    if (lastUpdateDatetimeInput) {
      lastUpdateDatetimeInput.value = convertDatetimeToLocaleString(detail.lastUpdateDatetime);
    }

    const datetimeInput = content.querySelector(`[data-${inst.tag}-detail='createDatetime']`)?.querySelector("input");
    if (datetimeInput) {
      datetimeInput.value = convertDatetimeToLocaleString(detail.createDatetime);
    }

    let fileName = detail.fileName ? detail.fileName : `${detail.messageId}_${detail.source}_response`;
    const fileExtension = getFileExtensionFromMimeType(detail.dataFormat);
    if (fileExtension) {
      fileName = fileName + `.${fileExtension}`;
    }
    const fileNameInput = content.querySelector(`[data-${inst.tag}-detail='fileName']`)?.querySelector("input");
    if (fileNameInput) {
      fileNameInput.value = fileName;
    }
    const downloadButton = content.querySelector(`[data-${inst.tag}-detail='download']`);
    if (downloadButton) {
      downloadButton.addEventListener('click', () => inst.store.dispatch(inst.actions.getFile(detail.messageId, fileName)));
    }
    // let params = {
    //   requestType: reqType,
    //   id: detail.id,
    //   filename: detail.originalFilename,
    // };
    // const downloadButton = content.querySelector(`[data-${inst.tag}-detail='download']`);
    // downloadButton.addEventListener('click', () => inst.store.dispatch(inst.actions.getFile(params)));

    // const datetimeInput = content.querySelector(`[data-${inst.tag}-detail='createDatetime']`).querySelector("input");
    // datetimeInput.value = convertDatetimeToLocaleString(detail.createDatetime);
  };
  return ret;
};
