"use strict";

import {handleCollapseCardHeaderOnClick} from './common.js';
import {handleDropDownOnClick, displayDropdownBoxLabel} from './common.js';
import {hasInvalidCharacter, isNumeric, isEmptyField} from './common.js';
import {checkRequiredFieldInItem, checkInvalidCharacterInItem} from './common.js';
import {handleInputLabels, handleModalOnToggleById, toggleLoadingIcon} from './common.js';
import {convertDatetimeToLocaleString, checkStatusColor, getFilterDateInputValue} from './common.js';
import {camelCaseToSentenceCase} from './common.js';
import {getHistoryTable} from './component/history_table.js';
import {getHistoryModal} from './component/history_modal.js';
import {exportFullRecordCSV, getReorderedHeadersAndRecArrs} from './component/history_export.js';
import './external/diff.min.js';

export const History = (store, actions, tabname) => {
  const inst = {};
  inst.store = store;
  inst.actions = actions;
  inst.tag = store.getState().tag;
  inst.store.subscribe((action) => {
    const {tab} = inst.store.getState();
    if (tabname && tab !== tabname) {
      return;
    }
    if (action.type === 'SET_RECORDS') {
      const {records} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='table']`);
      inst.table.buildTable(container, records);
    } else if (action.type === 'SET_TOTAL_PAGES') {
      const {totalPages, page, size} = inst.store.getState();
      const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='table']`);
      inst.table.buildPagination(container, page, size, totalPages);
      inst.table.buildExport(container, () => inst.store.dispatch(inst.actions.exportRecords(inst.onExportCompleted)));
    } else if (action.type === 'SET_TOTAL_RECORDS') {
      const {totalRecords} = inst.store.getState();
      inst.table.setTotalRecordCount(totalRecords);
    } else if (action.type === 'SET_RECORD_LOADING') {
      const {tab, recordLoading} = inst.store.getState();
      const table = document.querySelector(`[data-${inst.tag}-${tab}='table']`);
      const tableLoading = table?.querySelector("[data-table='loading']");
      toggleLoadingIcon(tableLoading, recordLoading);
    } else if (action.type === 'SET_EXPORT_LOADING') {
      const {tab, exportLoading} = inst.store.getState();
      const exportBtn = document.getElementById("history-export");
      exportBtn.disabled = exportLoading
      toggleLoadingIcon(exportBtn, exportLoading);
    }
  });
  inst.table = Object.create(getHistoryTable());
  inst.setTable = () => {
    inst.table.fillHistoryTableRow = function(cols, rec) {
      cols[0].textContent = convertDatetimeToLocaleString(rec.activityTime);
      const activityCategoryTemplate = document.getElementById("activity-category-span-template");
      const activityCategorySpan = activityCategoryTemplate.content.querySelector(`[data-${inst.tag}-span='${rec.activityCategory}']`)?.cloneNode(true);
      if (activityCategorySpan) {
        cols[1].appendChild(activityCategorySpan);
      } else {
        cols[1].textContent = rec.activityCategory;
      }

      const activityTypeTemplate = document.getElementById("activity-type-span-template");
      const activityTypeSpan = activityTypeTemplate.content.querySelector(`[data-${inst.tag}-span='${rec.activityType}']`)?.cloneNode(true);
      if (activityTypeSpan) {
        cols[2].appendChild(activityTypeSpan);
      } else {
        cols[2].textContent = rec.activityType;
      }

      const activityActionTypeTemplate = document.getElementById("activity-actiontype-span-template");
      const activityActionTypeSpan = activityActionTypeTemplate.content.querySelector(`[data-${inst.tag}-span='${rec.activityActionType}']`)?.cloneNode(true);
      if (activityActionTypeSpan) {
        cols[3].appendChild(activityActionTypeSpan);
      } else {
        cols[3].textContent = rec.activityActionType;
      }
      // Handle Status
      const statusSpan = document.createElement('span');
      const statusTextColor = checkStatusColor(rec.status);
      if (statusTextColor) {
        statusSpan.classList.add(`text-${statusTextColor}`);
      }
      statusSpan.innerHTML = rec.status;
      cols[4].appendChild(statusSpan);
      cols[5].textContent = rec.activityUsername;
      cols[6].textContent = rec.activityMessage;
      if (rec.activityMetaData && (
        rec.activityMetaData._originalData || rec.activityMetaData._newData)) {
        const viewButtonContainer = document.getElementById("activity-view-button-template").content.cloneNode(true);
        const viewButton = viewButtonContainer.querySelector("button");
        viewButton.addEventListener('click', () => inst.viewModal.handleHistoryModalOnToggle(rec.activityMetaData));
        cols[7].appendChild(viewButton);
      }
    };
    inst.table.setPage = function(page) {
      inst.store.dispatch(inst.actions.setPage(page))
    };
    inst.table.setSize = function(size) {
      inst.store.dispatch(inst.actions.setSize(size))
    };
  };
  inst.viewModal = Object.create(getHistoryModal());
  inst.setViewModal = () => {
    const modal = inst.viewModal
    const container = document.querySelector(`[data-${inst.tag}='content']`).querySelector("#detail-modal");
    modal.setContainer(container);
    const template = document.getElementById(`view-modal-content-template`);
    modal.setTemplate(template);
    modal.onActivate = function(activityMetaData) {
      inst.viewModal.buildModalContent(activityMetaData);
    };
    modal.renderModalContent = function(content, detail) {
      const originalDataText = JSON.stringify(detail._originalData ? detail._originalData : {}, null, '    ');
      const newDataText = JSON.stringify(detail._newData ? detail._newData : {}, null, '    ');
      let diff = Diff.diffLines(originalDataText, newDataText);
      console.log(diff);
      const renderLine = (container, line, lnum, state = null) => {
        const lineContainer = document.createElement("p");
        lineContainer.classList.add("w-fit");
        let lineLabelText;
        if (state === 'added') {
          lineContainer.classList.add("bg-green-400");
          lineLabelText = `       + | `;
        } else if (state === 'removed') {
          lineContainer.classList.add("bg-red-400");
          lineLabelText = `${String(lnum).padStart(6, ' ')} - | `;
        } else {
          lineLabelText = `${String(lnum).padStart(6, ' ')}   | `;
        }
        const lineLabel = document.createElement("span");
        lineLabel.textContent = lineLabelText;
        const lineContent = document.createElement("span");
        lineContent.textContent = line;
        lineContainer.appendChild(lineLabel);
        lineContainer.appendChild(lineContent);
        container.appendChild(lineContainer);
      };
      const originalDataContainer = content.querySelector(`[data-${inst.tag}-modal='originalDataContainer']`);
      originalDataText.split(/\r?\n/).forEach((line, idx) => renderLine(originalDataContainer, line, idx+1));
      const newDataContainer = content.querySelector(`[data-${inst.tag}-modal='newDataContainer']`);
      let lineNum = 1;
      diff.forEach((changeObj) => {
        let state = changeObj.added ? 'added' :
          changeObj.removed ? 'removed' : null;
        changeObj.value.split(/\r?\n/).forEach((line) => {
          if (!line.length) {
            return;
          }
          renderLine(newDataContainer, line, lineNum, state);
          if (changeObj.added) {
            return;
          }
          lineNum = lineNum + 1;
        });
      });
    };
    modal.renderModalFooter = function(container, detail) {};
  };
  inst.buildFilter = () => {
    const queryParams = ['activityCategory', 'activityType', 'activityActionType', 'status',
                         'activityUsername', 'activityTimeFrom', 'activityTimeTo'];
    const handleQueryInputOnChange = (name, target) => {
      let value;
      if (target.tagName === 'INPUT') {
        if (target.type === 'text') {
          value = target.value;
        } else if (target.type === 'checkbox') {
          value = target.checked;
        } else if (target.type === 'datetime-local' || target.type === 'date') {
          value = getFilterDateInputValue(name, target.value);
        }
      } else if (target.tagName === 'SELECT') {
        value = target.value;
      }
      const card = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='filter']`);
      const labelContainer = card.querySelector(`[data-${inst.tag}-filter='${name}']`);
      handleInputLabels(labelContainer, value, target.tagName);
      inst.store.dispatch(inst.actions.setQuery({[name]: value}));
    };
    const handleSearchButtonOnClick = () => {
      const {recordLoading} = inst.store.getState();
      if (recordLoading) {
        return;
      }
      if (!inst.isSearchButtonActive()) {
        return;
      }
      inst.store.dispatch(inst.actions.setPage());
    };
    const handleClearButtonOnClick = () => {
      const {recordLoading} = inst.store.getState();
      if (recordLoading) {
        return;
      }
      queryParams.forEach(q => {
        const fieldContainer = card.querySelector(`[data-${inst.tag}-filter='${q}']`);
        const inputs = Array.from(fieldContainer.querySelectorAll("input"));
        if (inputs && inputs.length) {
          inputs.forEach(input => {
            if (input.type === 'checkbox') {
              input.checked = false;
            } else {
              input.value = null;
            }
          });
        }
        const select = fieldContainer.querySelector("select");
        if (select) {
          select.selectedIndex = -1;
        }
      });
      const boxLabels = card.querySelectorAll("[data-dropdown='box']")
      boxLabels.forEach(elem => {
        elem.querySelector('p').innerHTML = "";
      });
      inst.store.dispatch(inst.actions.clearQuery());
      inst.store.dispatch(inst.actions.getRecords());
    };
    const buildCollapseCard = (card) => {
      const header = card.querySelector("[data-collapsecard='header']");
      header.addEventListener('click', () => handleCollapseCardHeaderOnClick(card));
    };
    const renderFilterContent = (card) => {
      const arrayParams = ['activityCategory', 'activityType', 'activityActionType', 'status'];
      queryParams.filter(q => !arrayParams.includes(q)).forEach(q => {
        const fieldContainer = card.querySelector(`[data-${inst.tag}-filter='${q}']`);
        const input = fieldContainer?.querySelector("input");
        if (input) {
          input.addEventListener("change", (e) => handleQueryInputOnChange(q, e.target));
        }
        const select = fieldContainer?.querySelector("select");
        if (select) {
          select.selectedIndex = -1;
          select.addEventListener("change", (e) => handleQueryInputOnChange(q, e.target));
        }
      });
      arrayParams.forEach(param => {
          const paramSelect = card.querySelector(`[data-${inst.tag}-filter='${param}']`);
          paramSelect.querySelector("[data-dropdown='box']").addEventListener('click', () => handleDropDownOnClick(paramSelect));
          paramSelect.querySelector("[data-dropdown='mask']").addEventListener('click', () => handleDropDownOnClick(paramSelect));
          const paramInputs = paramSelect.querySelector("[data-dropdown='container']").querySelectorAll('input');
          const paramTarget = () => {
            return {
              value: Array.from(paramInputs).filter(i => i.checked).map(o => o.value),
              tagName: 'SELECT',
            };
          };
          paramInputs.forEach(input => {
            input.addEventListener('change', () => displayDropdownBoxLabel(paramSelect));
            input.addEventListener('change', () => handleQueryInputOnChange(param, paramTarget()));
          });
      });
    };
    inst.isSearchButtonActive = () => {
      const {query} = store.getState();
      const checkInvalidCharacterFields = ['activityUsername'];
      let isActive = checkInvalidCharacterInItem(query, checkInvalidCharacterFields);
      return isActive;
    };
    const card = document.querySelector(`[data-${inst.tag}='content']`).querySelector(`[data-${inst.tag}-history='filter']`);
    buildCollapseCard(card);
    renderFilterContent(card);
    const searchButton = card.querySelector("#history-filter-search");
    searchButton.addEventListener("click", handleSearchButtonOnClick);
    const clearButton = card.querySelector("#history-filter-clear");
    clearButton.addEventListener("click", handleClearButtonOnClick);
  };
  inst.onExportCompleted = (fullRecords) => {
    // Customized export results handler
    // Get ActivityCategory, ActivityType, ActivityActionType Map
    const activityCategories = {};
    const activityTypes = {};
    const activityActionTypes = {};

    const activityCategoryTemplate = document.getElementById("activity-category-span-template");
    Array.from(activityCategoryTemplate.content.querySelectorAll(`[data-${inst.tag}-span]`)).forEach(elem => {
      activityCategories[elem.getAttribute(`data-${inst.tag}-span`)] =  elem.textContent;
    });
    const activityTypeTemplate = document.getElementById("activity-type-span-template")
    Array.from(activityTypeTemplate.content.querySelectorAll(`[data-${inst.tag}-span]`)).forEach(elem => {
      activityTypes[elem.getAttribute(`data-${inst.tag}-span`)] =  elem.textContent;
    });
    const activityActionTypeTemplate = document.getElementById("activity-actiontype-span-template")
    Array.from(activityActionTypeTemplate.content.querySelectorAll(`[data-${inst.tag}-span]`)).forEach(elem => {
      activityActionTypes[elem.getAttribute(`data-${inst.tag}-span`)] =  elem.textContent;
    });
    // Preprocess records before export
    fullRecords.forEach(rec => {
      let activityCategory = rec.activityCategory !== null && rec.activityCategory !== undefined ?
        activityCategories[rec.activityCategory] : undefined;
      if (activityCategory) {
        rec.activityCategory = activityCategory;
      }
      let activityType = rec.activityType !== null && rec.activityType !== undefined ?
        activityTypes[rec.activityType] : undefined;
      if (activityType) {
        rec.activityType = activityType;
      }
      let activityActionType = rec.activityActionType !== null && rec.activityActionType !== undefined ?
        activityActionTypes[rec.activityActionType] : undefined;
      if (activityActionType) {
        rec.activityActionType = activityActionType;
      }
    });
    // Export
    const order = ['id', 'activityCategory', 'activityType', 'activityActionType',
                   'status', 'activityUsername', 'activityTime', 'activityMessage']
    let {headers, recArrs} = getReorderedHeadersAndRecArrs(fullRecords, order)
    headers = headers.map(camelCaseToSentenceCase);
    // const headers = Object.keys(fullRecords[0])
    // const recArrs = fullRecords.map(Object.values);
    exportFullRecordCSV(`${inst.tag}_export.csv`, headers, recArrs);
  };
  inst.initialize = () => {
    const {tag} = inst.store.getState();
    inst.tag = tag;
    inst.buildFilter();
    inst.setTable();
    inst.setViewModal();
    store.dispatch(inst.actions.getRecords());
  };
  // Table related method
  return inst;
};
